<?php

declare(strict_types=1);

namespace Drupal\crowdsec;

use Drupal\Component\Plugin\PluginBase;
use Drupal\Core\Config\ImmutableConfig;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Base class for scenario plugins.
 */
abstract class ScenarioPluginBase extends PluginBase implements ContainerFactoryPluginInterface, ScenarioInterface {

  /**
   * Default settings for the plugin.
   *
   * @var array
   */
  protected static array $defaultSettings = [
    'enable' => TRUE,
    'leak_speed' => 10,
    'bucket_capacity' => 10,
    'ban_duration' => 3600,
  ];

  /**
   * The buffer service.
   *
   * @var \Drupal\crowdsec\Buffer
   */
  protected Buffer $buffer;

  /**
   * The Crowdsec settings.
   *
   * @var \Drupal\Core\Config\ImmutableConfig
   */
  protected ImmutableConfig $settings;

  /**
   * {@inheritdoc}
   */
  final public function __construct(array $configuration, string $plugin_id, array $plugin_definition, Buffer $buffer, ImmutableConfig $settings) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->buffer = $buffer;
    $this->settings = $settings;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): ScenarioInterface {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('crowdsec.buffer'),
      $container->get('config.factory')->get('crowdsec.settings'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function label(): string {
    // Cast the label to a string since it is a TranslatableMarkup object.
    return (string) $this->pluginDefinition['label'];
  }

  /**
   * {@inheritdoc}
   */
  final public function addSignal(string $ip, int $status, ?string $targetUser = NULL): void {
    if ($this->getPluginDefinition()['buffer']) {
      $this->buffer->bufferSignal($this, $ip, $status, $targetUser);
      return;
    }
    $context = [];
    $this->buffer->addCurrentContext($context, $status, $targetUser);
    $this->buffer->addSignal($this->getPluginDefinition()['scenario'], $ip, $this->getSetting('ban_duration'), $context);
  }

  /**
   * {@inheritdoc}
   */
  final public function getSetting(string $key): mixed {
    return $this->settings->get($this->getSettingKey($key)) ?? self::$defaultSettings[$key];
  }

  /**
   * {@inheritdoc}
   */
  final public function getSettingKey(string $key): string {
    if (!array_key_exists($key, self::$defaultSettings)) {
      throw new \InvalidArgumentException("Unknown setting key: $key");
    }
    return implode('.', ['plugins', $this->getPluginId(), $key]);
  }

  /**
   * {@inheritdoc}
   */
  final public function getStorageKey(string $ip, string $key): string {
    return implode('_', [$this->getPluginId() . '-' . $ip, $this->getPluginId(), $key]);
  }

}
