<?php

declare(strict_types=1);

namespace Drupal\crowdsec;

use CrowdSec\CapiClient\ClientException;
use Drupal\Component\Datetime\TimeInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\ImmutableConfig;
use Drupal\Core\KeyValueStore\KeyValueExpirableFactoryInterface;
use Drupal\Core\KeyValueStore\KeyValueFactoryInterface;
use Drupal\Core\KeyValueStore\KeyValueStoreExpirableInterface;
use Drupal\Core\KeyValueStore\KeyValueStoreInterface;
use Drupal\Core\Link;
use Drupal\Core\Lock\LockBackendInterface;
use Drupal\Core\Logger\LoggerChannelInterface;
use Drupal\Core\Queue\QueueFactory;
use Drupal\Core\Queue\QueueInterface;
use Drupal\Core\State\StateInterface;
use Drupal\Core\Url;
use Drupal\crowdsec\Event\CrowdSecEvents;
use Drupal\crowdsec\Event\IpBanned;
use Drupal\crowdsec\Event\IpSignalled;
use Symfony\Component\EventDispatcher\EventDispatcherInterface;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\RequestStack;

/**
 * Implements the CrowdSec buffer service.
 */
class Buffer {

  private const string LAST_SIGNAL_PUSH_TIMESTAMP = 'crowdsec.signal.push.timestamp';
  private const string SIGNALS = 'signals';
  private const string LOCK = 'crowdsec.signal';

  /**
   * The key-value store.
   *
   * @var \Drupal\Core\KeyValueStore\KeyValueStoreInterface
   */
  protected KeyValueStoreInterface $keyValue;

  /**
   * The expirable key-value store.
   *
   * @var \Drupal\Core\KeyValueStore\KeyValueStoreExpirableInterface
   */
  protected KeyValueStoreExpirableInterface $expirableKeyValue;

  /**
   * The CrowdSec configuration.
   *
   * @var \Drupal\Core\Config\ImmutableConfig
   */
  protected ImmutableConfig $config;

  /**
   * The queue.
   *
   * @var \Drupal\Core\Queue\QueueInterface
   */
  protected QueueInterface $queue;

  /**
   * The current request.
   *
   * @var \Symfony\Component\HttpFoundation\Request
   */
  protected Request $request;

  /**
   * Constructs the storage implementation.
   */
  public function __construct(
    protected StateInterface $state,
    protected KeyValueFactoryInterface $keyValueFactory,
    protected KeyValueExpirableFactoryInterface $keyValueExpirableFactory,
    protected LockBackendInterface $lock,
    protected TimeInterface $time,
    protected Client $client,
    protected LoggerChannelInterface $logger,
    protected ConfigFactoryInterface $configFactory,
    protected QueueFactory $queueFactory,
    protected EventDispatcherInterface $eventDispatcher,
    protected RequestStack $requestStack,
    protected Ban $banService,
  ) {
    $this->keyValue = $keyValueFactory->get('crowdsec');
    $this->expirableKeyValue = $keyValueExpirableFactory->get('crowdsec');
    $this->config = $configFactory->get('crowdsec.settings');
    $this->queue = $queueFactory->get('crowdsec', TRUE);
    $this->request = $this->requestStack->getCurrentRequest();
  }

  /**
   * Acquires a lock for CrowdSec to safely operate some non-concurrent actions.
   *
   * @return bool
   *   TRUE, if successful, FALSE otherwise.
   */
  private function lock(): bool {
    $attempts = 0;
    while (!$this->lock->acquire($this::LOCK)) {
      $attempts++;
      if ($attempts > 3) {
        return FALSE;
      }
      $this->lock->wait($this::LOCK, 1);
    }
    return TRUE;
  }

  /**
   * Adds the current context to the past context.
   *
   * @param array $context
   *   The past context.
   * @param int $status
   *   The response status for the current request.
   * @param string|null $targetUser
   *   The target user in case of a brute force attack.
   */
  public function addCurrentContext(array &$context, int $status, ?string $targetUser = NULL): void {
    $currentContext = [
      'target_uri' => $this->request->getRequestUri(),
      'user_agent' => $this->request->headers->get('User-Agent'),
      'method' => $this->request->getMethod(),
      'status' => $status,
    ];
    if ($targetUser !== NULL) {
      $currentContext['target_user'] = $targetUser;
    }
    foreach ($currentContext as $key => $value) {
      $context[] = [
        'key' => $key,
        'value' => $value,
      ];
    }
  }

  /**
   * Adds a new signal to the buffer for being sent later.
   *
   * @param string $scenario
   *   The scenario name.
   * @param string $ip
   *   The IP address to be blocked.
   * @param int $duration
   *   The duration in seconds for how long the IP should be blocked.
   * @param array $context
   *   The signal context.
   */
  public function addSignal(string $scenario, string $ip, int $duration, array $context): void {
    if (!in_array($scenario, $this->config->get('signal_scenarios'), TRUE)) {
      $this->logger->info('Ignoring signal for @ip because scenario @scenario is disabled.', [
        '@scenario' => $scenario,
        '@ip' => $ip,
      ]);
      return;
    }
    if (!$this->lock()) {
      // We can not acquire a lock and therefore can't buffer the new signal.
      $this->logger->critical('Can not acquire lock. Wanted to add signal to @scenario for @ip.', [
        '@scenario' => $scenario,
        '@ip' => $ip,
      ]);
      return;
    }
    $signals = $this->keyValue->get($this::SIGNALS, []);
    $signals[] = [
      'scenario' => $scenario,
      'ip' => $ip,
      'duration' => $duration,
      'timestamp' => $this->time->getRequestTime(),
      'context' => $context,
    ];
    $this->keyValue->set($this::SIGNALS, $signals);
    $this->lock->release($this::LOCK);
    $this->logger->warning('Buffered signal to @scenario for @ip.', [
      '@scenario' => $scenario,
      '@ip' => $ip,
      'link' => Link::fromTextAndUrl('Link', Url::fromUri(Client::CROWDSEC_URL_CTI . $ip))->toString(),
    ]);
    $this->eventDispatcher->dispatch(new IpSignalled($ip, $scenario), CrowdSecEvents::IP_SIGNALLED);
  }

  /**
   * Adds a new signal to the buffer.
   *
   * @param \Drupal\crowdsec\ScenarioInterface $plugin
   *   The scenario plugin.
   * @param string $ip
   *   The IP address.
   * @param int $status
   *   The response status code.
   * @param string|null $targetUser
   *   The optional target user.
   */
  public function bufferSignal(ScenarioInterface $plugin, string $ip, int $status, ?string $targetUser): void {
    if (!$plugin->getSetting('enable')) {
      $this->logger->info('Ignoring @scenario signal for @ip because it is disabled.', [
        '@scenario' => $plugin->getPluginDefinition()['id'],
        '@ip' => $ip,
      ]);
      return;
    }
    $duration = $plugin->getSetting('ban_duration');
    $leakSpeed = $plugin->getSetting('leak_speed');
    $bucketCapacity = $plugin->getSetting('bucket_capacity');
    $requestTime = $this->time->getRequestTime();

    $bucketFill = $this->expirableKeyValue->get($plugin->getStorageKey($ip, 'count'), 0);
    $lastTime = $this->expirableKeyValue->get($plugin->getStorageKey($ip, 'latest_time'), $requestTime);
    $context = $this->expirableKeyValue->get($plugin->getStorageKey($ip, 'context'), []);
    $this->addCurrentContext($context, $status, $targetUser);
    $bucketFill++;
    $bucketFill -= floor(($requestTime - $lastTime) / $leakSpeed);

    if ($bucketFill > $bucketCapacity) {
      // Threshold reached, take actions.
      // - ban the ip.
      $this->banService->banIp($ip);
      $this->logger->info('Banned ip @ip', [
        '@ip' => $ip,
        'link' => Link::fromTextAndUrl('Link', Url::fromUri(Client::CROWDSEC_URL_CTI . $ip))->toString(),
      ]);
      $this->eventDispatcher->dispatch(new IpBanned($ip), CrowdSecEvents::IP_BANNED);
      // - add task to unban the ip.
      $this->queue->createItem([
        'type' => 'unban',
        'ip' => $ip,
        'due' => $requestTime + $duration,
      ]);
      // - add signal.
      $this->addSignal($plugin->getPluginDefinition()['scenario'], $ip, $duration, $context);
    }

    if ($bucketFill <= 0) {
      // Delete history.
      $this->expirableKeyValue->delete($plugin->getStorageKey($ip, 'count'));
      $this->expirableKeyValue->delete($plugin->getStorageKey($ip, 'latest_time'));
      $this->expirableKeyValue->delete($plugin->getStorageKey($ip, 'context'));
    }
    else {
      // Threshold not reached yet, update history.
      $expire = (int) (2 * $leakSpeed * $bucketFill);
      $this->expirableKeyValue->setWithExpire($plugin->getStorageKey($ip, 'count'), $bucketFill, $expire);
      $this->expirableKeyValue->setWithExpire($plugin->getStorageKey($ip, 'latest_time'), $requestTime, $expire);
      $this->expirableKeyValue->setWithExpire($plugin->getStorageKey($ip, 'context'), $context, $expire);
    }
  }

  /**
   * Push buffered signals to CrowdSec.
   */
  public function push(): void {
    $lastPush = $this->state->get($this::LAST_SIGNAL_PUSH_TIMESTAMP, 0);
    $now = $this->time->getCurrentTime();
    if ($lastPush + 10 > $now) {
      // It's too early, wait for the next round.
      return;
    }
    $watcher = $this->client->watcher();
    $pushSignals = [];
    if (!$this->lock()) {
      // We can not acquire a lock and therefore can't safely push.
      $this->logger->critical('Can not acquire lock. Wanted to push signals upstream.');
      return;
    }
    $signals = $this->keyValue->get($this::SIGNALS, []);
    $i = 0;
    while ($signal = array_shift($signals)) {
      $datetime = new \DateTime();
      $datetime
        ->setTimezone(new \DateTimeZone('UTC'))
        ->setTimestamp($signal['timestamp']);
      try {
        $pushSignals[] = $watcher->buildSignal([
          'scenario' => $signal['scenario'],
          'created_at' => $datetime,
          'message' => '',
          'context' => $signal['context'] ?? [],
        ], [
          'value' => $signal['ip'],
        ], [
          [
            'duration' => $signal['duration'],
          ],
        ]);
      }
      catch (ClientException $e) {
        $this->logger->error('Can not create signal on scenario @scenario for IP @ip: @message', [
          '@scenario' => $signal['scenario'],
          '@ip' => $signal['ip'],
          '@message' => $e->getMessage(),
        ]);
      }
      $i++;
      if ($i > 250) {
        break;
      }
    }
    if (!empty($pushSignals)) {
      $watcher->pushSignals($pushSignals);
      $this->logger->notice('Pushed @count signals upstream.<br><hr>Signal Details:<br>@details', [
        '@count' => $i,
        '@details' => implode('<br>', array_map(function ($signal) {
          return json_encode($signal);
        }, $pushSignals)),
      ]);
    }
    $this->keyValue->set($this::SIGNALS, $signals);
    $this->lock->release($this::LOCK);
    $this->state->set($this::LAST_SIGNAL_PUSH_TIMESTAMP, $now);
  }

}
