<?php

declare(strict_types=1);

namespace Drupal\crowdsec\EventSubscriber;

use Drupal\Core\EventSubscriber\HttpExceptionSubscriberBase;
use Drupal\crowdsec\Buffer;
use Drupal\crowdsec\Client;
use Drupal\crowdsec\ScenarioPluginManager;
use Drupal\user\Event\UserEvents;
use Drupal\user\Event\UserFloodEvent;
use Symfony\Component\HttpKernel\Event\ExceptionEvent;

/**
 * CrowdSec event subscriber.
 */
class CrowdSec extends HttpExceptionSubscriberBase {

  /**
   * Constructs the CrowdSec event subscriber service.
   */
  public function __construct(
    protected Client $client,
    protected Buffer $buffer,
  ) {}

  /**
   * {@inheritdoc}
   */
  protected static function getPriority(): int {
    // Get work done even before Fast404.
    return 205;
  }

  /**
   * {@inheritdoc}
   */
  protected function getHandledFormats(): array {
    return ['html'];
  }

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents(): array {
    $events = parent::getSubscribedEvents();
    $events[UserEvents::FLOOD_BLOCKED_IP] = ['onBlockIp'];
    $events[UserEvents::FLOOD_BLOCKED_USER] = ['onBlockIp'];
    return $events;
  }

  /**
   * Handles all 4xx errors for HTML.
   *
   * @param \Symfony\Component\HttpKernel\Event\ExceptionEvent $event
   *   The event to process.
   */
  public function on4xx(ExceptionEvent $event): void {
    $ip = $event->getRequest()->getClientIp();
    if (empty($ip)) {
      return;
    }
    /** @var \Symfony\Component\HttpKernel\Exception\HttpExceptionInterface $exception */
    $exception = $event->getThrowable();
    ScenarioPluginManager::getPlugin('whisper')->addSignal($ip, $exception->getStatusCode());
  }

  /**
   * Will be called when flood control bans an IP.
   *
   * @param \Drupal\user\Event\UserFloodEvent $event
   *   The user flood event.
   */
  public function onBlockIp(UserFloodEvent $event): void {
    $targetUser = $event->getName() === UserEvents::FLOOD_BLOCKED_USER ? (string) $event->getUid() : NULL;
    ScenarioPluginManager::getPlugin('flood')->addSignal($event->getIp(), 403, $targetUser);
  }

}
