<?php

namespace Drupal\crowdsec;

use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\crowdsec\Event\CrowdSecEvents;
use Drupal\crowdsec\Event\ScenarioList;
use Drupal\crowdsec\Event\SignalScenarioList;
use Symfony\Component\EventDispatcher\EventDispatcherInterface;

/**
 * Implementation of the CrowdSec storage interface.
 */
class Signals {

  use StringTranslationTrait;

  /**
   * Constructs the signals service.
   */
  public function __construct(
    protected readonly EventDispatcherInterface $eventDispatcher,
    protected readonly ScenarioPluginManager $scenarioPluginManager,
  ) {}

  /**
   * Computes all valid choices for the "signal_scenarios" setting.
   *
   * @see crowdsec.schema.yml
   *
   * @return string[]
   *   All valid choices.
   */
  public static function validSignalScenarios(): array {
    /** @var \Drupal\crowdsec\Signals $signals */
    $signals = \Drupal::service('crowdsec.signals');
    return array_keys($signals->signalScenarios());
  }

  /**
   * Computes all valid choices for the "scenarios" setting.
   *
   * @see crowdsec.schema.yml
   *
   * @return string[]
   *   All valid choices.
   */
  public static function validScenarios(): array {
    /** @var \Drupal\crowdsec\Signals $signals */
    $signals = \Drupal::service('crowdsec.signals');
    return array_keys($signals->scenarios());
  }

  /**
   * Provides a list of available scenarios that can be signaled upstream.
   *
   * @return array
   *   The available scenarios for upstream signalling.
   */
  public function signalScenarios(): array {
    $scenarios = [];
    foreach ($this->scenarioPluginManager->getDefinitions() as $definition) {
      $scenarios[$definition['scenario']] = $definition['label'];
    }
    // Allow other modules to add more scenarios.
    $this->eventDispatcher->dispatch(new SignalScenarioList($scenarios), CrowdSecEvents::SIGNAL_SCENARIO_LIST_BUILD);
    return $scenarios;
  }

  /**
   * Provides a list of available scenarios.
   *
   * @return array
   *   The available scenarios.
   */
  public function scenarios(): array {
    $scenarios = $this->signalScenarios() + [
      'crowdsecurity/http-backdoors-attempts' => $this->t('Detect attempt to common backdoors'),
      'crowdsecurity/http-bad-user-agent' => $this->t('Detect bad user-agents'),
      'crowdsecurity/http-crawl-non_statics' => $this->t('Detect aggressive crawl from single ip'),
      'crowdsecurity/http-probing' => $this->t('Detect site scanning/probing from a single ip'),
      'crowdsecurity/http-path-traversal-probing' => $this->t('Detect path traversal attempt'),
      'crowdsecurity/http-sensitive-files' => $this->t('Detect attempt to access to sensitive files (.log, .db ..) or folders (.git)'),
      'crowdsecurity/http-sqli-probing' => $this->t('A scenario that detects SQL injection probing with minimal false positives'),
      'crowdsecurity/http-xss-probing' => $this->t('A scenario that detects XSS probing with minimal false positives'),
      'crowdsecurity/http-w00tw00t' => $this->t('Detect w00tw00t'),
      'crowdsecurity/http-generic-bf' => $this->t('Detect generic http brute force'),
      'crowdsecurity/http-open-proxy' => $this->t('Detect scan for open proxy'),
    ];
    // Allow other modules to add more scenarios.
    $this->eventDispatcher->dispatch(new ScenarioList($scenarios), CrowdSecEvents::SCENARIO_LIST_BUILD);
    return $scenarios;
  }

}
