<?php

declare(strict_types=1);

namespace Drupal\crowdsec\Plugin\QueueWorker;

use Drupal\Component\Datetime\TimeInterface;
use Drupal\Core\Link;
use Drupal\Core\Logger\LoggerChannelInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Queue\QueueFactory;
use Drupal\Core\Queue\QueueInterface;
use Drupal\Core\Queue\QueueWorkerBase;
use Drupal\Core\Url;
use Drupal\crowdsec\Ban;
use Drupal\crowdsec\Client;
use Drupal\crowdsec\Event\CrowdSecEvents;
use Drupal\crowdsec\Event\IpUnBanned;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\EventDispatcher\EventDispatcherInterface;

/**
 * CrowdSec queue worker.
 *
 * @QueueWorker(
 *   id = "crowdsec",
 *   title = @Translation("CrowdSec"),
 *   cron = {"time" = 3}
 * )
 */
final class CrowdSec extends QueueWorkerBase implements ContainerFactoryPluginInterface {

  /**
   * The queue.
   *
   * @var \Drupal\Core\Queue\QueueInterface
   */
  protected QueueInterface $queue;

  /**
   * Constructs the CrowdSec queue worker.
   */
  public function __construct(
    array $configuration,
    string $plugin_id,
    array $plugin_definition,
    protected LoggerChannelInterface $logger,
    protected QueueFactory $queueFactory,
    protected TimeInterface $time,
    protected EventDispatcherInterface $eventDispatcher,
    protected Ban $banService,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->queue = $queueFactory->get('crowdsec', TRUE);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): CrowdSec {
    return new CrowdSec(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('logger.channel.crowdsec'),
      $container->get('queue'),
      $container->get('datetime.time'),
      $container->get('event_dispatcher'),
      $container->get('crowdsec.ban'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function processItem($data): void {
    if ($data['type'] === 'unban' && $data['due'] <= $this->time->getRequestTime()) {
      $this->banService->unbanIp($data['ip']);
      $this->logger->info('Un-banned ip @ip', [
        '@ip' => $data['ip'],
        'link' => Link::fromTextAndUrl('Link', Url::fromUri(Client::CROWDSEC_URL_CTI . $data['ip']))->toString(),
      ]);
      $this->eventDispatcher->dispatch(new IpUnBanned($data['ip']), CrowdSecEvents::IP_UNBANNED);
      return;
    }

    // Item not processed, let's re-queue.
    $this->queue->createItem($data);
  }

}
