<?php

namespace Drupal\crux\EventSubscriber;

use Drupal\ckeditor_mentions\Events\CKEditorEvents;
use Drupal\ckeditor_mentions\Events\CKEditorMentionsEvent;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Component\Datetime\TimeInterface;
use Drupal\Core\Logger\LoggerChannelInterface;
use Drupal\Core\Queue\QueueFactory;
use Drupal\user\EntityOwnerInterface;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Symfony\Contracts\EventDispatcher\EventDispatcherInterface;
use Drupal\Core\State\StateInterface;
use Drupal\crux\Event\CruxEvents;
use Drupal\crux\Event\CruxGlobalDailyLimitExceededEvent;
use Drupal\crux\Event\CruxUserDailyMentionLimitExceededEvent;

/**
 * Subscribes to mention events to enqueue Crux AI responses.
 */
class CruxMentionSubscriber implements EventSubscriberInterface {

  public function __construct(
    protected QueueFactory $queueFactory,
    protected ConfigFactoryInterface $configFactory,
    protected LoggerChannelInterface $logger,
    protected TimeInterface $time,
    protected StateInterface $state,
    protected EventDispatcherInterface $eventDispatcher,
  ) {}

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents(): array {
    return [
      CKEditorEvents::MENTION_FIRST => 'onFirstMention',
    ];
  }

  /**
   * React to first-time mention events.
   */
  public function onFirstMention(CKEditorMentionsEvent $event): void {
    $config = $this->configFactory->get('crux.settings');
    $bot_uid = (int) $config->get('bot_uid');

    $ai_provider_id = (string) $config->get('ai_provider_id');
    if (!$ai_provider_id) {
      $this->logger->warning('Crux mention detected but no AI provider configured; cannot respond.');
      return;
    }

    $mentioned = $event->getMentionedEntity();
    if ($mentioned->getEntityTypeId() !== 'user') {
      return;
    }
    if ($bot_uid === 0 || (int) $mentioned->id() !== $bot_uid) {
      // Not the Crux bot.
      return;
    }

    // Need a source that has an author.
    $source = $event->getEntity();
    if (!$source || !$source instanceof EntityOwnerInterface) {
      return;
    }

    // Determine author (content author who initiated the mention).
    $author_id = (int) $source->getOwnerId();

    // Load / initialise daily counters.
    $today = date('Ymd', $this->time->getRequestTime());
    $counts = $this->state->get('crux.daily_counts', []);
    if (empty($counts['date']) || $counts['date'] !== $today) {
      $counts = [
        'date' => $today,
        'global' => 0,
        'users' => [],
      ];
    }

    $global_limit = (int) $config->get('global_daily_limit');
    $user_limit = (int) $config->get('user_daily_mention_limit');

    $current_global = (int) ($counts['global'] ?? 0);
    $current_user = $author_id ? (int) ($counts['users'][$author_id] ?? 0) : 0;

    // Enforce global limit.
    if ($global_limit > 0 && $current_global >= $global_limit) {
      $this->logger->notice('Crux global daily mention limit (@limit) reached; mention ignored.', ['@limit' => $global_limit]);
      $this->eventDispatcher->dispatch(
        new CruxGlobalDailyLimitExceededEvent($global_limit, $current_global, $source),
        CruxEvents::GLOBAL_DAILY_LIMIT_EXCEEDED,
      );
      return;
    }

    // Enforce per-user limit (skip if unlimited).
    if ($user_limit > 0 && $current_user >= $user_limit) {
      $this->logger->notice('Crux user daily mention limit (@limit) reached for user @uid; mention ignored.', [
        '@limit' => $user_limit,
        '@uid' => $author_id,
      ]);
      $this->eventDispatcher->dispatch(
        new CruxUserDailyMentionLimitExceededEvent($author_id, $user_limit, $current_user, $source),
        CruxEvents::USER_DAILY_LIMIT_EXCEEDED,
      );
      return;
    }

    $item = [
      'entity_type' => $source->getEntityTypeId(),
      'entity_id' => $source->id(),
      'throttle_after' => $this->time->getRequestTime() + ((int) $config->get('throttle_response_time') * 60),
    ];

    $this->queueFactory->get('crux_mentions_response')->createItem($item);

    // Update counters and persist.
    $counts['global'] = $current_global + 1;
    $counts['users'][$author_id] = $current_user + 1;
    $this->state->set('crux.daily_counts', $counts);
  }

}
