<?php

namespace Drupal\crux\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\ai\Service\AiProviderFormHelper;
use Drupal\ai\AiProviderPluginManager;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Configure Crux settings.
 */
class CruxSettingsForm extends ConfigFormBase {

  /**
   * Operation type used for Crux responses.
   *
   * Using 'chat' as the default AI operation type.
   */
  protected const CRUX_OPERATION_TYPE = 'chat';

  /**
   * AI form helper.
   */
  protected AiProviderFormHelper $aiFormHelper;

  /**
   * AI provider manager.
   */
  protected AiProviderPluginManager $providerManager;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    /** @var \Drupal\ai\Service\AiProviderFormHelper $ai_form_helper */
    $ai_form_helper = $container->get('ai.form_helper');
    /** @var \Drupal\ai\AiProviderPluginManager $provider_manager */
    $provider_manager = $container->get('ai.provider');
    /** @var \Drupal\crux\Form\CruxSettingsForm $instance */
    $instance = parent::create($container);
    $instance->aiFormHelper = $ai_form_helper;
    $instance->providerManager = $provider_manager;
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'crux_settings_form';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array {
    return ['crux.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $config = $this->config('crux.settings');

    $form['processing'] = [
      '#type' => 'details',
      '#title' => $this->t('Processing limits'),
      '#open' => TRUE,
    ];

    $form['processing']['user_daily_mention_limit'] = [
      '#type' => 'number',
      '#title' => $this->t('User daily mention limit'),
      '#default_value' => $config->get('user_daily_mention_limit') ?? 3,
      '#min' => 0,
      '#description' => $this->t('Maximum number of times a single content author can trigger the bot per day. Set 0 for unlimited.'),
      '#required' => TRUE,
    ];

    $form['processing']['global_daily_limit'] = [
      '#type' => 'number',
      '#title' => $this->t('Global daily limit'),
      '#default_value' => $config->get('global_daily_limit') ?? 50,
      '#min' => 0,
      '#description' => $this->t('Maximum total number of bot responses queued per day across the site. Set 0 for unlimited.'),
      '#required' => TRUE,
    ];

    $form['processing']['throttle_response_time'] = [
      '#type' => 'number',
      '#title' => $this->t('Throttle response time (minutes)'),
      '#default_value' => $config->get('throttle_response_time') ?? 5,
      '#min' => 0,
      '#description' => $this->t('Minimum number of minutes to wait after the mention before generating a response (helps batch rapid edits).'),
      '#required' => TRUE,
    ];

    // Build provider form inside ai details using helper. Prefix 'crux'.
    $provider_id = $config->get('ai_provider_id') ?? '';
    $model_id = $config->get('ai_model_id') ?? '';

    // We call helper which appends elements to the passed form array.
    $this->aiFormHelper->generateAiProvidersForm(
      $form,
      $form_state,
      self::CRUX_OPERATION_TYPE,
      'crux',
      AiProviderFormHelper::FORM_CONFIGURATION_FULL,
      0,
      '',
      $this->t('AI Provider'),
      $this->t('Choose the AI provider to generate Crux responses.'),
      TRUE,
    );

    // Set defaults if present in config and not yet in the element.
    if (isset($form['crux_ai_provider']) && empty($form_state->getUserInput()['crux_ai_provider'])) {
      $form['crux_ai_provider']['#default_value'] = $provider_id ?: $form['crux_ai_provider']['#default_value'];
    }
    if (isset($form['crux_ajax_prefix']['crux_ai_model']) && empty($form_state->getUserInput()['crux_ai_model'])) {
      $form['crux_ajax_prefix']['crux_ai_model']['#default_value'] = $model_id ?: $form['crux_ajax_prefix']['crux_ai_model']['#default_value'];
    }

    $form['cleanup'] = [
      '#type' => 'details',
      '#title' => $this->t('Uninstall cleanup'),
      '#open' => FALSE,
      '#description' => $this->t('Control whether all content and the Crux bot user are deleted when the module is uninstalled.'),
    ];

    $form['cleanup']['delete_on_uninstall'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Delete all Crux content and user on uninstall'),
      '#default_value' => (bool) $config->get('delete_on_uninstall'),
      '#description' => $this->t('If checked, all nodes and comments authored by the Crux bot plus the bot user account will be permanently deleted during uninstall. Leave unchecked to preserve historical content.'),
    ];

    // System prompt (advanced) fieldset.
    $form['ai_prompt'] = [
      '#type' => 'details',
      '#title' => $this->t('Default AI system prompt'),
      '#description' => $this->t('Controls baseline behavior and tone for Crux. Typically you should not edit this unless you need to significantly change how Crux responds.'),
      '#open' => FALSE,
    ];

    $form['ai_prompt']['default_system_prompt'] = [
      '#type' => 'textarea',
      '#title' => $this->t('System prompt'),
      '#default_value' => $config->get('default_system_prompt') ?? '',
      '#rows' => 10,
      '#description' => $this->t('Primary instruction sent to the AI provider. Keep concise and avoid secrets or PII.'),
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state): void {
    parent::validateForm($form, $form_state);
    // Delegate validation of provider configuration.
    $this->aiFormHelper->validateAiProvidersConfig($form, $form_state, self::CRUX_OPERATION_TYPE, 'crux');
    // Ensure a provider & model are selected (allowing '__default__').
    if (!$form_state->getValue('crux_ai_provider')) {
      $form_state->setErrorByName('crux_ai_provider', $this->t('You must select an AI provider.'));
    }
    if ($form_state->getValue('crux_ai_provider') && $form_state->getValue('crux_ai_provider') !== '__default__' && !$form_state->getValue('crux_ai_model')) {
      $form_state->setErrorByName('crux_ai_model', $this->t('You must select an AI model.'));
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $provider_id = (string) $form_state->getValue('crux_ai_provider');
    $model_id = (string) $form_state->getValue('crux_ai_model');

    // Build dynamic provider configuration.
    $provider_configuration = [];
    if ($provider_id && $provider_id !== '__default__' && $model_id) {
      $provider_configuration = $this->aiFormHelper->generateAiProvidersConfigurationFromForm($form, $form_state, self::CRUX_OPERATION_TYPE, 'crux');
    }

    $this->configFactory->getEditable('crux.settings')
      ->set('user_daily_mention_limit', (int) $form_state->getValue('user_daily_mention_limit'))
      ->set('global_daily_limit', (int) $form_state->getValue('global_daily_limit'))
      ->set('throttle_response_time', (int) $form_state->getValue('throttle_response_time'))
      ->set('ai_provider_id', $provider_id)
      ->set('ai_model_id', $model_id)
      ->set('ai_provider_configuration', $provider_configuration)
      ->set('default_system_prompt', (string) $form_state->getValue('default_system_prompt'))
      ->save();

    parent::submitForm($form, $form_state);
  }

}
