<?php

namespace Drupal\crux\Plugin\QueueWorker;

use Drupal\ai\AiProviderPluginManager;
use Drupal\ai\OperationType\Chat\ChatInput;
use Drupal\ai\OperationType\Chat\ChatMessage;
use Drupal\comment\CommentInterface;
use Drupal\comment\CommentManagerInterface;
use Drupal\Component\Datetime\TimeInterface;
use Drupal\Component\Utility\Html;
use Drupal\Component\Uuid\UuidInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Logger\LoggerChannelInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Queue\DelayedRequeueException;
use Drupal\Core\Queue\QueueWorkerBase;
use League\CommonMark\CommonMarkConverter;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Processes queued Crux mention responses.
 *
 * Creates an AI generated reply to a comment mention and, if the comment
 * field supports threading, attaches the new comment as a child (pid) of the
 * original comment. Adds parent (primary) entity context (title + body
 * excerpt) so the AI model can answer with better specificity.
 * For non-threaded comment fields, the immediately previous comment (if any)
 * on the same entity is used as lightweight conversational context.
 *
 * @QueueWorker(
 *   id = "crux_mentions_response",
 *   title = @Translation("Crux mention responses"),
 *   cron = {"time" = 30}
 * )
 */
class CruxMentionsResponse extends QueueWorkerBase implements ContainerFactoryPluginInterface {

  /**
   * Constructs a new CruxMentionsResponse queue worker.
   */
  public function __construct(
    array $configuration,
    $plugin_id,
    $plugin_definition,
    protected EntityTypeManagerInterface $entityTypeManager,
    protected LoggerChannelInterface $logger,
    protected TimeInterface $time,
    protected ConfigFactoryInterface $configFactory,
    protected AiProviderPluginManager $aiProviderManager,
    protected UuidInterface $uuid,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): static {
    /** @var \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager */
    $entity_type_manager = $container->get('entity_type.manager');
    /** @var \Drupal\Core\Logger\LoggerChannelInterface $logger */
    $logger = $container->get('logger.channel.crux');
    /** @var \Drupal\Component\Datetime\TimeInterface $time */
    $time = $container->get('datetime.time');
    /** @var \Drupal\Core\Config\ConfigFactoryInterface $config_factory */
    $config_factory = $container->get('config.factory');
    /** @var \Drupal\ai\AiProviderPluginManager $ai_provider */
    $ai_provider = $container->get('ai.provider');
    /** @var \Drupal\Component\Uuid\UuidInterface $uuid */
    $uuid = $container->get('uuid');

    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $entity_type_manager,
      $logger,
      $time,
      $config_factory,
      $ai_provider,
      $uuid,
    );
  }

  /**
   * {@inheritdoc}
   */
  public function processItem($data): void {
    // Throttle: if current time is before throttle_after, requeue later.
    $now = $this->time->getRequestTime();
    if (isset($data['throttle_after']) && $data['throttle_after'] > $now) {
      throw new DelayedRequeueException(
        $data['throttle_after'] - $now,
        'Crux response throttled; will retry later.'
      );
    }

    $entity = NULL;
    if (!empty($data['entity_type']) && !empty($data['entity_id'])) {
      $storage = $this->entityTypeManager->getStorage($data['entity_type']);
      /** @var \Drupal\user\EntityOwnerInterface $entity */
      $entity = $storage->load($data['entity_id']);
    }

    if (!$entity) {
      $this->logger->warning(
        'Crux queue item references missing entity @type:@id',
        [
          '@type' => $data['entity_type'] ?? 'unknown',
          '@id' => $data['entity_id'] ?? 'n/a',
        ]
      );
      return;
    }

    $mention_text = '';
    $parent_entity = NULL;

    // Most common case: entity is a comment (the mention itself).
    if ($entity instanceof CommentInterface) {
      $parent_entity = $entity->getCommentedEntity();

      // If the comment has comment_body, assume it's the mention source.
      if ($entity->hasField('comment_body') && !$entity->get('comment_body')->isEmpty()) {
        $mention_text = strip_tags($entity->get('comment_body')->value);
      }
    }
    else {
      // Not a comment; only comments supported for auto-replies.
      $this->logger->notice(
        'Crux mention auto-reply skipped: entity @type:@id is not a comment.',
        [
          '@type' => $entity->getEntityTypeId(),
          '@id' => $entity->id(),
        ]
      );
      return;
    }

    // Load Crux settings.
    $config = $this->configFactory->get('crux.settings');
    $provider_id = $config->get('ai_provider_id');
    $model_id = $config->get('ai_model_id');
    if (!$provider_id || !$model_id) {
      $this->logger->error(
        'Crux AI configuration incomplete: provider (@provider) or model (@model) missing.',
        [
          '@provider' => $provider_id ?? 'none',
          '@model' => $model_id ?? 'none',
        ]
      );
      return;
    }

    /** @var \Drupal\ai\AiProviderInterface $provider */
    $provider = $this->aiProviderManager->createInstance($provider_id);
    $providerConfig = $config->get('ai_provider_configuration') ?? [];
    $provider->setConfiguration($providerConfig);

    // Build extended context (primary entity title + body excerpt) used as
    // input to the AI.
    $primary_context = '';
    if ($parent_entity) {
      $title = trim((string) $parent_entity->label());
      if ($title !== '') {
        $primary_context .= "Primary content title: \n" . $title . "\n\n";
      }
      if ($parent_entity->hasField('body') && !$parent_entity->get('body')->isEmpty()) {
        $body_raw = (string) $parent_entity->get('body')->value;
        $body_plain = strip_tags($body_raw);
        $body_excerpt = function_exists('mb_substr')
          ? mb_substr($body_plain, 0, 200)
          : substr($body_plain, 0, 200);
        if ($body_excerpt !== '') {
          $primary_context .= "Primary content excerpt (first 200 chars):\n";
          $primary_context .= $body_excerpt . "\n\n";
        }
      }
    }

    // Determine threading capabilities early for context building.
    $threaded = FALSE;
    if ($entity instanceof CommentInterface && $parent_entity) {
      $comment_field = $entity->getFieldName();
      $field_config_id = $parent_entity->getEntityTypeId() . '.' . $parent_entity->bundle() . '.' . $comment_field;
      /** @var \Drupal\field\FieldConfigInterface|null $field_config */
      $field_config = $this->entityTypeManager->getStorage('field_config')->load($field_config_id);
      if ($field_config) {
        $settings = $field_config->getSettings();
        $default_mode = (int) ($settings['default_mode'] ?? 0);
        $threaded = $default_mode === CommentManagerInterface::COMMENT_MODE_THREADED;
      }
    }

    // Build message list including up to 3 parent thread comments.
    // Order: oldest to newest for conversational clarity.
    $messages = [];
    if ($threaded) {
      $parent_messages = $this->buildParentThreadMessages($entity, 3);
      if ($parent_messages) {
        $messages = array_merge($messages, $parent_messages);
      }
    }
    else {
      // For non-threaded fields include previous flat comment as context.
      $previous_message = $this->buildPreviousFlatCommentMessage($entity);
      if ($previous_message) {
        $messages[] = $previous_message;
      }
    }

    $context = '';
    if ($primary_context !== '') {
      $context .= $primary_context;
    }
    $context .= "Mention content:\n";
    $context .= ($mention_text !== '' ? $mention_text : '[No textual body captured]') . "\n\n";
    $messages[] = new ChatMessage('user', $context);

    $input = new ChatInput($messages);
    $input->setSystemPrompt($config->get('default_system_prompt'));

    try {
      $response = $provider->chat($input, $model_id, ['crux'])->getNormalized();
    }
    catch (\Exception $e) {
      $this->logger->error('Crux AI provider error: @message', ['@message' => $e->getMessage()]);
      return;
    }

    $converter = new CommonMarkConverter();
    $reply_html = (string) $converter->convert($response->getText());

    // Inline author mention.
    $bot_uid = (int) $config->get('bot_uid');
    $reply_html = $this->addAuthorMentionInline($reply_html, $entity, $bot_uid);

    // Create and save reply comment (re-uses $threaded computed earlier).
    $this->createReplyComment($entity, $reply_html, $bot_uid, $threaded);
  }

  /**
   * Adds a CKEditor mention anchor for the original comment author inline.
   *
   * If the generated HTML begins with a <p> tag, the anchor is inserted as the
   * first inline element inside that paragraph; otherwise it's prepended.
   * Skips if author is missing or is the bot user.
   */
  protected function addAuthorMentionInline(string $reply_html, CommentInterface $original_comment, int $bot_uid): string {
    $author = $original_comment->getOwner();
    if (!$author || !$author->id() || $author->id() === $bot_uid) {
      return $reply_html;
    }

    $mention_anchor = '<a class="mention"'
      . ' data-mention="@' . $author->id() . '"'
      . ' data-mention-uuid="' . $this->uuid->generate() . '"'
      . ' data-entity-type="user"'
      . ' data-entity-uuid="' . $author->uuid() . '"'
      . ' data-plugin="user"'
      . ' href="' . $author->toUrl()->toString() . '">@'
      . Html::escape($author->getDisplayName()) . '</a>';

    $trimmed = ltrim($reply_html);
    if (str_starts_with($trimmed, '<p')) {
      $count = 0;
      $modified = preg_replace(
        '/<p(\b[^>]*)>/',
        '<p$1>' . $mention_anchor . ' ',
        $reply_html,
        1,
        $count
      );
      if ($count > 0 && $modified !== NULL) {
        return $modified;
      }
    }
    return $mention_anchor . ' ' . $reply_html;
  }

  /**
   * Builds ChatMessages for up to $max parent (ancestor) comments.
   *
   * Collected oldest first so conversation order is preserved.
   */
  protected function buildParentThreadMessages(CommentInterface $comment, int $max = 3): array {
    $messages = [];
    $ancestors = [];
    $current = $comment;
    $loaded = 0;
    $storage = $this->entityTypeManager->getStorage('comment');

    while ($loaded < $max) {
      $pid = (int) ($current->get('pid')->target_id ?? 0);
      if (!$pid) {
        break;
      }
      /** @var \Drupal\comment\CommentInterface|null $parent */
      $parent = $storage->load($pid);
      if (!$parent instanceof CommentInterface) {
        break;
      }
      $ancestors[] = $parent;
      $current = $parent;
      $loaded++;
    }

    if (!$ancestors) {
      return [];
    }

    // Reverse to chronological order: oldest ancestor first.
    $ancestors = array_reverse($ancestors);

    // Determine bot uid to map roles properly (bot authored => assistant).
    $bot_uid = (int) ($this->configFactory->get('crux.settings')->get('bot_uid') ?? 0);

    foreach ($ancestors as $ancestor) {
      $author = $ancestor->getOwner();
      $author_name = $author?->getDisplayName() ?? 'Unknown user';
      $body_value = '';
      if ($ancestor->hasField('comment_body') && !$ancestor->get('comment_body')->isEmpty()) {
        $body_value = strip_tags($ancestor->get('comment_body')->value);
        if (function_exists('mb_substr')) {
          $body_value = mb_substr($body_value, 0, 400);
        }
        else {
          $body_value = substr($body_value, 0, 400);
        }
      }
      $text = "Previous comment by {$author_name}:\n" . ($body_value !== '' ? $body_value : '[No body]');
      $role = ($author && (int) $author->id() === $bot_uid) ? 'assistant' : 'user';
      $messages[] = new ChatMessage($role, $text);
    }

    return $messages;
  }

  /**
   * Builds a ChatMessage for the previous flat (non-threaded) comment.
   *
   * Previous comment determined by created timestamp (< current).
   * Returns NULL if none found.
   */
  protected function buildPreviousFlatCommentMessage(CommentInterface $comment): ?ChatMessage {
    $parent_entity = $comment->getCommentedEntity();
    if (!$parent_entity) {
      return NULL;
    }
    $comment_field = $comment->getFieldName();

    $storage = $this->entityTypeManager->getStorage('comment');
    $query = $storage->getQuery()
      ->condition('entity_type', $parent_entity->getEntityTypeId())
      ->condition('entity_id', $parent_entity->id())
      ->condition('field_name', $comment_field)
      ->condition('created', $comment->getCreatedTime(), '<')
      ->sort('created', 'DESC')
      ->range(0, 1)
      ->accessCheck(FALSE);

    $cids = $query->execute();
    if (!$cids) {
      return NULL;
    }
    $previous = $storage->load(reset($cids));
    if (!$previous instanceof CommentInterface) {
      return NULL;
    }

    $author = $previous->getOwner();
    $author_name = $author?->getDisplayName() ?? 'Unknown user';
    $body_value = '';
    if ($previous->hasField('comment_body') && !$previous->get('comment_body')->isEmpty()) {
      $body_value = strip_tags($previous->get('comment_body')->value);
      if (function_exists('mb_substr')) {
        $body_value = mb_substr($body_value, 0, 400);
      }
      else {
        $body_value = substr($body_value, 0, 400);
      }
    }

    $bot_uid = (int) ($this->configFactory->get('crux.settings')->get('bot_uid') ?? 0);
    $role = ($author && (int) $author->id() === $bot_uid) ? 'assistant' : 'user';
    $text = "Previous comment (flat) by {$author_name}:\n" . ($body_value !== '' ? $body_value : '[No body]');
    return new ChatMessage($role, $text);
  }

  /**
   * Creates and saves the reply comment entity.
   *
   * Handles threading (pid) when requested. The parent entity is derived
   * from the original comment's commented entity. Safely aborts if required
   * context is missing.
   */
  protected function createReplyComment(CommentInterface $original_comment, string $reply_html, int $bot_uid, bool $threaded): void {
    $parent_entity = $original_comment->getCommentedEntity();
    if (!$parent_entity) {
      $this->logger->warning('Skipping reply creation: parent entity missing for comment @id.', ['@id' => $original_comment->id()]);
      return;
    }

    $comment_field = $original_comment->getFieldName();
    $comment_type = $original_comment->bundle();

    $comment_values = [
      'entity_type' => $parent_entity->getEntityTypeId(),
      'entity_id' => $parent_entity->id(),
      'field_name' => $comment_field,
      'uid' => $bot_uid,
      'comment_type' => $comment_type,
      'subject' => 'Re: ' . ($original_comment->getSubject() ?? 'Your mention'),
      'comment_body' => [
        'value' => $reply_html,
        'format' => 'full_html',
      ],
      'status' => 1,
    ];

    if ($threaded) {
      $comment_values['pid'] = $original_comment->id();
    }

    $comment = $this->entityTypeManager->getStorage('comment')->create($comment_values);
    $comment->save();
  }

}
