const { PurgeCSS } = require('purgecss');
const fs = require('fs');
const path = require('path');
const glob = require('glob');

// --- 1. CONFIGURATION ---
const themeDir = __dirname;

// STRICT INPUT: Only looking at this specific file
const inputCssPath = path.join(themeDir, 'bootstrap.min.css');

// OUTPUT: Where the cleaned file will go (Root of theme)
const outputCssPath = path.join(themeDir, 'bootstrap-purge.min.css');

// CSV: Your safelist file
const csvPath = path.join(themeDir, 'css-classes.csv');

// CONTENT: Where to look for class usage
const contribDir = path.resolve(__dirname, '../../contrib/bootstrap_barrio');
const searchPatterns = [
  path.join(themeDir, 'templates/**/*.html.twig'),
  path.join(themeDir, 'js/**/*.js'),
  path.join(contribDir, '**/*.twig'),
];

async function runPurge() {
  console.log('\n========================================');
  console.log('🚀 SINGLE FILE PURGE: bootstrap.min.css');
  console.log('========================================');

  // --- STEP A: VERIFY INPUT EXISTS ---
  if (!fs.existsSync(inputCssPath)) {
    console.error('❌ CRITICAL ERROR: Input file not found at:');
    console.error(`   ${inputCssPath}`);
    return; // Stop the script
  }
  console.log('✅ Input File Found: css/vendor/bootstrap.min.css');

  // --- STEP B: LOAD CSV SAFELIST ---
  let drupalClasses = [];
  try {
    if (fs.existsSync(csvPath)) {
      const fileContent = fs.readFileSync(csvPath, 'utf-8');
      drupalClasses = fileContent
        .replace(/\r?\n/g, ',')
        .split(',')
        .map((item) => item.trim().replace(/['"]/g, '').replace(/^\./, ''))
        .filter((item) => item.length > 1 && item !== 'class_name');

      console.log(`✅ CSV Safelist: Loaded ${drupalClasses.length} classes.`);
    }
  } catch (err) {
    console.error('❌ Error reading CSV:', err);
  }

  // --- STEP C: SCAN CONTENT FILES ---
  let contentFiles = [];
  searchPatterns.forEach((p) => {
    contentFiles = contentFiles.concat(glob.sync(p));
  });
  console.log(`✅ Content Scanned: ${contentFiles.length} files (Twig/JS).`);

  // --- STEP D: EXECUTE PURGECSS ---
  console.log('⏳ Processing... (Holding strict safelist)');

  const purgeResult = await new PurgeCSS().purge({
    content: contentFiles,

    // STRICTLY ONE FILE
    css: [inputCssPath],

    defaultExtractor: (content) => (
      content.match(/([A-Za-z0-9_-]+)|(?<=['"])(.*?)(?=['"])/g) || []
    ),

    safelist: [
      // 1. Dynamic Regex
      /^col-/, /^btn-/, /^modal-/, /^fade/, /^show/, /^nav-/, /^navbar-/, /container/,
      // 2. CSV Classes
      ...drupalClasses,
    ],
  });

  // --- STEP E: WRITE OUTPUT ---
  if (purgeResult[0]) {
    const { css } = purgeResult[0];
    fs.writeFileSync(outputCssPath, css);
    console.log('\n🎉 DONE! Output saved to: ./bootstrap.min.css');

    // Final check for your specific missing class
    if (css.includes('.mt-5') || css.includes('mt-5')) {
      console.log('✅ VERIFICATION: \'mt-5\' is present in the output.');
    } else {
      console.error('❌ VERIFICATION FAILED: \'mt-5\' is missing.');
      console.error('   (Note: Check if \'mt-5\' exists in css/vendor/bootstrap.min.css)');
    }
  } else {
    console.error('❌ No CSS was generated.');
  }
  console.log('========================================\n');
}

runPurge();
