<?php

namespace Drupal\css_class_collector\Batch;

use Drupal\Core\Database\Database;
use Drupal\views\Views;

class Scanner {

  /**
   * Save unique CSS classes.
   */
  public static function saveClasses(array $classes) {
    $conn = Database::getConnection();
    foreach ($classes as $class) {
      try {
        $conn->insert('css_class_collector')
          ->fields(['class_name' => $class])
          ->execute();
      }
      catch (\Exception $e) {
        // Duplicate, ignore.
      }
    }
  }

  /**
   * Extract CSS classes using the service.
   */
  public static function extract($html) {
    return \Drupal::service('css_class_collector.extractor')->extractClasses($html);
  }

  /**
   * NODE
   */
  public static function processNode($nid, &$context) {
    $node = \Drupal\node\Entity\Node::load($nid);
    if (!$node) return;

    $classes = self::scanEntity($node);
    self::saveClasses($classes);

    $context['results']['nodes'][] = $nid;
  }

  /**
   * PARAGRAPH
   */
  public static function processParagraph($pid, &$context) {
    $paragraph = \Drupal\paragraphs\Entity\Paragraph::load($pid);
    if (!$paragraph) return;

    $classes = self::scanEntity($paragraph);
    self::saveClasses($classes);

    $context['results']['paragraphs'][] = $pid;
  }

  /**
   * BLOCK CONTENT
   */
  public static function processBlock($bid, &$context) {
    $block = \Drupal\block_content\Entity\BlockContent::load($bid);
    if (!$block) return;

    $classes = self::scanEntity($block);
    self::saveClasses($classes);

    $context['results']['blocks'][] = $bid;
  }

  /**
   * Shared entity scanner (same as your hook).
   */
  private static function scanEntity($entity) {
    $all = [];

    foreach ($entity->getFields() as $field) {
      $type = $field->getFieldDefinition()->getType();

      if (in_array($type, ['string_long', 'text_long', 'text_with_summary'])) {
        foreach ($field as $item) {
          $all = array_merge($all, self::extract($item->value));
        }
      }
    }

    return array_unique($all);
  }

  /**
   * SCAN ALL VIEWS
   */
  public static function processViews(&$context) {
    $views = Views::getAllViews();

    foreach ($views as $view) {
      $view->setDisplay($view->getDisplay()->display['id']);
      $render = $view->preview();

      $renderer = \Drupal::service('renderer');
      $html = $renderer->renderRoot($render);

      $classes = self::extract($html);
      self::saveClasses($classes);
    }

    $context['results']['views'] = 'done';
  }

  /**
   * FINISH
   */
  public static function finishBatch($success, $results, $operations) {
    if ($success) {
      \Drupal::messenger()->addMessage('Full scan completed successfully.');
    }
    else {
      \Drupal::messenger()->addError('Some operations failed.');
    }
  }
}
