<?php

namespace Drupal\css_variables_customizer;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Plugin\Component;
use Drupal\Core\TempStore\PrivateTempStoreFactory;
use Drupal\css_variables_customizer\EventSubscriber\CssVariablesPreviewCleanupSubscriber;

/**
 * Manages variables.
 */
class CssVariablesManager implements CssVariablesManagerInterface {

  /**
   * The private tempstore factory.
   *
   * @var \Drupal\Core\TempStore\PrivateTempStoreFactory
   */
  protected PrivateTempStoreFactory $tempStoreFactory;

  /**
   * The CSS variables preview cleanup subscriber.
   *
   * @var \Drupal\css_variables_customizer\EventSubscriber\CssVariablesPreviewCleanupSubscriber
   */
  protected CssVariablesPreviewCleanupSubscriber $cleanupSubscriber;

  /**
   * Constructs the variable manager.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   Used to get customizations.
   * @param \Drupal\Core\TempStore\PrivateTempStoreFactory $temp_store_factory
   *   The private tempstore factory.
   * @param \Drupal\css_variables_customizer\EventSubscriber\CssVariablesPreviewCleanupSubscriber $cleanup_subscriber
   *   The CSS variables preview cleanup subscriber.
   */
  public function __construct(
    protected ConfigFactoryInterface $configFactory,
    PrivateTempStoreFactory $temp_store_factory,
    CssVariablesPreviewCleanupSubscriber $cleanup_subscriber,
  ) {
    $this->tempStoreFactory = $temp_store_factory;
    $this->cleanupSubscriber = $cleanup_subscriber;
  }

  /**
   * {@inheritdoc}
   */
  public function getFileVariables(string $file) : array {
    $variables = [];
    $new_variables = $this->parseCss(file_get_contents($file));
    foreach ($new_variables as $category => $category_variables) {
      $variables[$category] = array_merge($variables[$category] ?? [], $category_variables);
    }
    return $variables;
  }

  /**
   * Parse css file contents.
   *
   * @param string $css_file_contents
   *   File contents.
   *
   * @return array
   *   Css variables grouped by category.
   */
  protected function parseCss(string $css_file_contents) : array {
    $categories = [];

    $current_selector = NULL;
    $current_category = NULL;
    $css_lines = explode(PHP_EOL, $css_file_contents);
    foreach ($css_lines as $line) {
      if (preg_match("/(?<selector>[:,a-z-\.\#\s\(\)]+) {/", $line, $selector_matches)) {
        $current_selector = !empty($current_selector) ? $current_selector . ' ' . $selector_matches['selector'] : $selector_matches['selector'];
      }
      elseif (preg_match(self::CSS_CATEGORY_START_REGEXP, $line, $category_regexp_matches) && !empty($category_regexp_matches['category'])) {
        $current_category = trim($category_regexp_matches['category']);
      }
      elseif (preg_match(self::CSS_CATEGORY_END_REGEXP, $line)) {
        $current_category = NULL;
      }
      elseif (preg_match(self::CSS_VARIABLES_REGEXP, $line, $variables_regexp_matches) && !empty($variables_regexp_matches['variable']) && !empty($current_category)
      && !empty($variables_regexp_matches['value'])) {
        $categories[$current_category][$variables_regexp_matches['variable']] = [
          'value' => trim($variables_regexp_matches['value']),
          'selector' => $current_selector,
        ];
      }
    }

    return array_filter($categories);
  }

  /**
   * {@inheritdoc}
   */
  public function getStringVariables(string $css_styles) : array {
    preg_match_all(self::CSS_VARIABLES_REGEXP, $css_styles, $variables_regexp_matches);

    return array_combine($variables_regexp_matches['variable'] ?? [], $variables_regexp_matches['value'] ?? []);
  }

  /**
   * {@inheritdoc}
   */
  public function variablesToStyle(array $variables, ?string $selector = NULL) : string {
    $customizations_style = '';
    $styles_by_selector = [];
    foreach ($variables as $variable => $variants) {
      foreach ($variants as $value) {
        $variable_selector = !empty($selector) ? $selector . ' ' : $value['selector'];
        $styles_by_selector[$variable_selector][] = sprintf('%s: %s;', $variable, $value['value']);
      }
    }

    foreach ($styles_by_selector as $selector => $css) {
      $customizations_style .= $selector . '{ ' . implode(PHP_EOL, $css) . '} ';
    }

    return trim($customizations_style);
  }

  /**
   * {@inheritdoc}
   */
  public function getComponentVariables(Component $component) : array {
    $component_css_files = [];
    $component_path = str_replace(DRUPAL_ROOT . '/', '', $component->getPluginDefinition()['path']);
    foreach ($component->library['css'] ?? [] as $css_group) {
      $files = array_keys($css_group);
      $files = array_filter(array_map(function ($file) use ($component_path) {
        preg_match(sprintf('#.*%s/(?<path>.*)#', $component_path), $file, $theme_matches);
        return !empty($theme_matches['path']) ? $component_path . '/' . $theme_matches['path'] : NULL;
      }, $files));
      $component_css_files = array_merge($component_css_files, $files);
    }

    $variables = [];
    foreach ($component_css_files as $file) {
      $variables = array_merge($variables, $this->getFileVariables($file));
    }
    return $variables;
  }

  /**
   * {@inheritdoc}
   */
  public function buildThemeCustomizations(string $theme, ?string $selector = NULL) : array {
    $customizations_render_array = [];

    // Check for preview data in tempstore first.
    $temp_store = $this->tempStoreFactory->get('css_variables_customizer');
    $preview_data = $temp_store->get('preview_' . $theme);

    if ($preview_data) {
      // Use preview data from tempstore.
      $customizations = $preview_data['customizations'] ?? [];
      $custom_variables = $preview_data['custom'] ?? [];

      // Mark preview data for cleanup after response is sent.
      $this->cleanupSubscriber->markForCleanup($theme);
    }
    else {
      // Use saved configuration.
      $config = $this->configFactory->get('css_variables_customizer.customizations.' . $theme);
      $customizations = $config->get('customizations') ?? [];
      $custom_variables = $config->get('custom') ?? [];
    }

    // Process customizations.
    foreach ($customizations as $customization) {
      if (!empty($customization['overrides'])) {
        $id = 'css_variables_customizer_' . $customization['id'];
        $customizations_render_array[$id] = [
          '#type' => 'html_tag',
          '#tag' => 'style',
          '#value' => $this->variablesToStyle($customization['overrides'], $selector),
          '#cache' => [
            'tags' => ['config:css_variables_customizer.customizations.' . $theme],
            'contexts' => ['css_variables_preview:' . $theme],
          ],
        ];
      }
    }

    // Process custom variables.
    if (!empty($custom_variables)) {
      $custom_variables_formatted = [];
      foreach ($custom_variables as $variable => $value) {
        $custom_variables_formatted[$variable] = [
          [
            'selector' => ':root',
            'value' => $value,
          ],
        ];
      }

      $customizations_render_array['css_variables_customizer_custom'] = [
        '#type' => 'html_tag',
        '#tag' => 'style',
        '#value' => $this->variablesToStyle($custom_variables_formatted, $selector),
        '#cache' => [
          'tags' => ['config:css_variables_customizer.customizations.' . $theme],
          'contexts' => ['css_variables_preview:' . $theme],
        ],
      ];
    }

    // Add empty CSS variables customizer style block to remove the
    // needing on clearing caches when overriding variables
    // of a cached page / component.
    if (empty($customizations_render_array)) {
      $customizations_render_array['css_variables_customizer_holder'] = [
        '#type' => 'html_tag',
        '#tag' => 'style',
        '#value' => '/* CSS variables customizer holder. */',
        '#cache' => [
          'tags' => ['config:css_variables_customizer.customizations.' . $theme],
          'contexts' => ['css_variables_preview:' . $theme],
        ],
      ];
    }

    return $customizations_render_array;
  }

}
