<?php

namespace Drupal\css_variables_customizer;

use Drupal\Core\Plugin\Component;

/**
 * Process CSS files and raw content to override existing CSS variables.
 */
interface CssVariablesManagerInterface {

  const CSS_CATEGORY_START_REGEXP = '#/\* @css-variables-customizer-category\s*(?<category>\w+)\s*\*/#';
  const CSS_CATEGORY_END_REGEXP = '#/\* @css-variables-customizer-category-end \*/#';
  const CSS_VARIABLES_REGEXP = '/(?<variable>--([a-zA-Z0-9]+-?)+):(?<value>.+);/';

  /**
   * Get the variables from a list of files.
   *
   * @param string $file
   *   The file.
   */
  public function getFileVariables(string $file) : array;

  /**
   * Parse css file contents.
   *
   * @param string $css_styles
   *   File contents.
   *
   * @return array
   *   Css variables grouped by category.
   */
  public function getStringVariables(string $css_styles) : array;

  /**
   * Convert a list of variables to a css style string.
   *
   * @param array $variables
   *   List of variables. It must have this structure:
   *    [
   *      variable_name => [
   *        [
   *          selector => selector,
   *          value => value,
   *        ]
   *      ],
   *    ].
   * @param string $selector
   *   Optional selector to override configuration selector.
   *
   * @return string
   *   The css style string.
   */
  public function variablesToStyle(array $variables, ?string $selector = NULL) : string;

  /**
   * Get the variables for a component.
   *
   * @param \Drupal\Core\Theme\ComponentPluginInterface $component
   *   The component.
   *
   * @return array
   *   The variables.
   */
  public function getComponentVariables(Component $component) : array;

  /**
   * Builds a list of render arrays with the theme customizations.
   *
   * @param string $theme
   *   Theme we want its customizations.
   * @param string $selector
   *   Optional selector to override configuration selector.
   *
   * @return array
   *   Each element is a <style> tag.
   */
  public function buildThemeCustomizations(string $theme, ?string $selector = NULL) : array;

}
