<?php

namespace Drupal\css_variables_customizer\EventSubscriber;

use Drupal\Core\TempStore\PrivateTempStoreFactory;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Symfony\Component\HttpKernel\Event\TerminateEvent;
use Symfony\Component\HttpKernel\KernelEvents;

/**
 * Cleans up CSS variables preview data after response is sent.
 */
class CssVariablesPreviewCleanupSubscriber implements EventSubscriberInterface {

  /**
   * The private tempstore factory.
   *
   * @var \Drupal\Core\TempStore\PrivateTempStoreFactory
   */
  protected PrivateTempStoreFactory $tempStoreFactory;

  /**
   * The themes to clean up preview data for.
   *
   * @var array
   */
  protected array $themesToCleanup = [];

  /**
   * Constructs a CssVariablesPreviewCleanupSubscriber object.
   *
   * @param \Drupal\Core\TempStore\PrivateTempStoreFactory $temp_store_factory
   *   The private tempstore factory.
   */
  public function __construct(PrivateTempStoreFactory $temp_store_factory) {
    $this->tempStoreFactory = $temp_store_factory;
  }

  /**
   * Mark a theme for cleanup after response.
   *
   * @param string $theme
   *   The theme name to cleanup preview data for.
   */
  public function markForCleanup(string $theme): void {
    if (!in_array($theme, $this->themesToCleanup, TRUE)) {
      $this->themesToCleanup[] = $theme;
    }
  }

  /**
   * Cleans up preview data after response is sent.
   *
   * @param \Symfony\Component\HttpKernel\Event\TerminateEvent $event
   *   The terminate event.
   */
  public function onTerminate(TerminateEvent $event): void {
    if (!empty($this->themesToCleanup)) {
      $temp_store = $this->tempStoreFactory->get('css_variables_customizer');
      foreach ($this->themesToCleanup as $theme) {
        $temp_store->delete('preview_' . $theme);
      }
      // Clear the array for next request.
      $this->themesToCleanup = [];
    }
  }

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents(): array {
    return [
      KernelEvents::TERMINATE => ['onTerminate', 0],
    ];
  }

}
