<?php

namespace Drupal\css_variables_customizer;

use Drupal\Core\Plugin\Component;
use Drupal\Core\Url;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Allows previewing components through integration with SDC Styleguide module.
 */
trait SdcStyleguidePreviewTrait {

  /**
   * Used to use SDC styleguide only if it is available.
   *
   * @var \Symfony\Component\DependencyInjection\ContainerInterface
   */
  protected ContainerInterface $container;

  /**
   * Returns TRUE when sdc styleguide module is available.
   */
  protected function sdcStyleguideAvailable() : bool {
    return $this->container->has('sdc_styleguide.demo_manager');
  }

  /**
   * Get the list of SDC demos for a specific component.
   *
   * @param \Drupal\Core\Plugin\Component $component
   *   Component.
   */
  protected function getSdcDemosForComponent(Component $component) : array {
    if (!$this->sdcStyleguideAvailable()) {
      return [];
    }

    /** @var \Drupal\sdc_styleguide\Service\SDCDemoManager $demoManager */
    $demoManager = $this->container->get('sdc_styleguide.demo_manager');
    $allDemos = $demoManager->getDemos();

    $componentId = $component->getPluginId();
    $definition = $component->getPluginDefinition();
    $group = $definition['group'] ?? 'Ungrouped';

    // Check if demos exist for this component.
    if (isset($allDemos[$group][$componentId]['demos'])) {
      return $allDemos[$group][$componentId]['demos'];
    }

    return [];
  }

  /**
   * Get the URL of a specific component demo.
   *
   * @param \Drupal\Core\Plugin\Component $component
   *   Component.
   * @param string $demo
   *   Demo we want to show.
   */
  protected function getSdcDemoUrl(Component $component, string $demo) : Url {
    $componentId = $component->getPluginId();
    $definition = $component->getPluginDefinition();
    $group = $definition['group'] ?? 'Ungrouped';

    // Build the URL using the sdc_styleguide.viewer route.
    return Url::fromRoute('sdc_styleguide.viewer', [
      'group' => $group,
      'component' => $componentId,
      'demo' => $demo,
    ]);
  }

}
