<?php

namespace Drupal\css_variables_customizer\Form;

use Drupal\Component\Render\FormattableMarkup;
use Drupal\Component\Utility\Html;
use Drupal\Component\Utility\NestedArray;
use Drupal\Core\Extension\Extension;
use Drupal\Core\Extension\ModuleExtensionList;
use Drupal\Core\Extension\ThemeExtensionList;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Render\Element as RenderElement;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\Theme\ComponentPluginManager;
use Drupal\css_variables_customizer\CssVariablesManagerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Plugin\Component;
use Drupal\css_variables_customizer\SdcStyleguidePreviewTrait;
use Drupal\Core\TempStore\PrivateTempStore;

/**
 * Customizer form.
 */
class CustomizerForm extends ConfigFormBase {

  use StringTranslationTrait;
  use SdcStyleguidePreviewTrait;

  /**
   * The relative path where the theme is located.
   *
   * Used to get manifest and extract css variables.
   *
   * @var string
   */
  protected string $themePath;

  /**
   * Used to get theme assets.
   *
   * @var \Drupal\Core\Extension\ThemeExtensionList
   */
  protected ThemeExtensionList $themeExtension;

  /**
   * Used to get component provider name.
   *
   * @var \Drupal\Core\Extension\ModuleExtensionList
   */
  protected ModuleExtensionList $moduleExtension;

  /**
   * Used to override theme components variables.
   *
   * @var \Drupal\Core\Theme\ComponentPluginManager
   */
  protected ComponentPluginManager $componentPluginManager;

  /**
   * Used to read CSS vars.
   *
   * @var \Drupal\css_variables_customizer\CssVariablesManagerInterface
   */
  protected CssVariablesManagerInterface $cssVariablesManager;

  /**
   * Used to store editable config names.
   *
   * @var array
   */
  protected array $editableConfigNames = [];

  /**
   * The private tempstore.
   *
   * @var \Drupal\Core\TempStore\PrivateTempStore
   */
  protected PrivateTempStore $tempStore;

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'css_variables_customizer.customizer';
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    $instance = parent::create($container);
    $instance->themeExtension = $container->get('extension.list.theme');
    $instance->moduleExtension = $container->get('extension.list.module');
    $instance->componentPluginManager = $container->get('plugin.manager.sdc');
    $instance->cssVariablesManager = $container->get('css_variables_customizer.manager');
    $instance->container = $container;
    $instance->tempStore = $container->get('tempstore.private')->get('css_variables_customizer');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function getEditableConfigNames() {
    return $this->editableConfigNames;
  }

  /**
   * Setup dynamic editable config names.
   */
  protected function setupEditableConfigNames(FormStateInterface $form_state) {
    $theme = $form_state->get('theme');
    if (!empty($theme)) {
      $this->editableConfigNames = [$this->getThemeConfigName($theme)];
    }
  }

  /**
   * Get the name of the configuration of the current theme.
   *
   * @param string $theme
   *   Theme being configured.
   */
  protected function getThemeConfigName(string $theme) {
    return 'css_variables_customizer.customizations.' . $theme;
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $theme = $this->getRouteMatch()->getParameter('theme');
    $theme_extension = $this->themeExtension->get($theme);

    $form_state->set('theme', $theme);

    $this->setupEditableConfigNames($form_state);

    $config = $this->config($this->getThemeConfigName($theme));

    $form['#attached']['library'][] = 'css_variables_customizer/css-variables-customizer-form';

    $form['customizations'] = [
      '#type' => 'vertical_tabs',
      '#title' => $this->t('CSS variables'),
      '#description' => $this->t('Adjust each theme available option of your choice. List is automatically generated based into your current css variables / properties, add more to expose here.'),
      '#open' => TRUE,
      '#tree' => TRUE,
    ];

    $main_css_assets = $this->getMainCssAssets($theme_extension);
    $main_css_variables = [];
    foreach ($main_css_assets as $main_css_asset) {
      $main_css_variables = array_merge($main_css_variables, $this->cssVariablesManager->getFileVariables($main_css_asset));
    }

    $form['customizations']['global'] = $this->createCustomizationsGroup('global', $this->t('Global'), $main_css_variables, $form_state);

    $components = $this->componentPluginManager->getAllComponents();

    uasort($components, function (Component $component_a, Component $component_b) use ($theme) {
      return $this->sortComponentsCallback($component_a, $component_b, $theme);
    });

    foreach ($components as $component) {
      $components_variables = $this->cssVariablesManager->getComponentVariables($component);
      if (!empty($components_variables)) {
        $plugin_definition = $component->getPluginDefinition();
        $form['customizations'][$component->getPluginId()] = $this->createCustomizationsGroup(
          $component->getPluginId(),
          $this->getProviderName($plugin_definition['provider']) . ': ' . $plugin_definition['name'],
          $components_variables,
          $form_state,
          $component
        );
      }
    }

    $custom_variables = $config->get('custom') ?? [];
    $custom_default_value = '';
    foreach ($custom_variables as $variable => $value) {
      $custom_default_value .= $variable . ':' . $value . ';' . PHP_EOL;
    }
    $form['customizations']['custom'] = [
      '#type' => 'details',
      '#title' => $this->t('Custom'),
      '#group' => 'customizations',
      'variables' => [
        '#type' => 'textarea',
        '#title' => $this->t('Custom CSS Variables'),
        '#description' => $this->t('Add new variables here, separated by new lines. Example: --theme-primary: #000000;'),
        '#default_value' => $custom_default_value,
        '#weight' => -999,
      ],
    ];

    $form['#validate'][] = [$this, 'validate'];
    return parent::buildForm($form, $form_state);
  }

  /**
   * Get the name of a component provider (module / theme).
   *
   * @param string $provider
   *   Provider.
   *
   * @return string
   *   Name of the module or theme. If not found, the self provider name.
   */
  protected function getProviderName(string $provider) {
    if ($this->themeExtension->exists($provider)) {
      return $this->themeExtension->get($provider)->info['name'];
    }
    elseif ($this->moduleExtension->exists($provider)) {
      return $this->moduleExtension->get($provider)->info['name'];
    }

    return $provider;
  }

  /**
   * Sorts the components so current theme assets are shown first.
   *
   * @param \Drupal\Core\Plugin\Component $component_a
   *   Fist component.
   * @param \Drupal\Core\Plugin\Component $component_b
   *   Second component.
   * @param string $theme
   *   Theme.
   */
  protected function sortComponentsCallback(Component $component_a, Component $component_b, string $theme) {
    $component_a_provider = $component_a->getPluginDefinition()['provider'];
    $component_b_provider = $component_b->getPluginDefinition()['provider'];
    if ($component_a_provider == $theme && $component_b_provider != $theme) {
      return -1;
    }
    elseif ($component_b_provider == $theme && $component_a_provider != $theme) {
      return 1;
    }
    return strcmp($component_a->getPluginDefinition()['name'], $component_b->getPluginDefinition()['name']);
  }

  /**
   * Create a customization fieldset to edit its variables.
   *
   * @param string $key
   *   The key of the customization.
   * @param string $label
   *   The label of the customization.
   * @param array $variables
   *   The variables of the customization.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   Form state.
   * @param \Drupal\Core\Plugin\Component|null $component
   *   The component object if available.
   *
   * @return array
   *   The form element.
   */
  protected function createCustomizationsGroup(string $key, string $label, array $variables, FormStateInterface $form_state, ?Component $component = NULL) {
    $form_element = [
      '#type' => 'details',
      '#title' => $label,
      '#group' => 'customizations',
    ];

    // Add Preview field with AJAX button if component is provided.
    if ($component instanceof Component && $this->sdcStyleguideAvailable()) {
      $preview = $this->buildComponentPreview($component, $key);
      if ($preview) {
        $form_element['preview'] = $preview;
      }
    }

    $form_element['preview']['#prefix'] = '<div class="css-variables-customizer-customization-group">';

    $customization = $this->getCustomizationsForKey($key, $form_state);

    $selector = $key == 'global' ? ':root' : sprintf(':root [data-component-id="%s"]', $key) ?? NULL;

    $form_element['categories'] = [
      '#type' => 'container',
      '#suffix' => '</div>',
      '#attributes' => [
        'class' => [
          'css-variables-customizer-categories',
        ],
      ],
    ];
    foreach ($variables as $category => $category_variables) {
      $form_element['categories'][$category] = [
        '#title' => ucfirst($category),
        '#type' => 'details',
      ];

      foreach ($category_variables as $variable => $value) {
        $this->addVariableFormElement($form_element['categories'], $category, $selector, $variable, $value, $customization['overrides'][$variable] ?? [], $form_state);
      }
    }

    $children = RenderElement::children($form_element['categories']);
    if (count($children) == 1) {
      $first_group = reset($children);
      $form_element['categories'][$first_group]['#open'] = TRUE;
    }
    return $form_element;
  }

  /**
   * Build component preview form element.
   *
   * @param \Drupal\Core\Plugin\Component $component
   *   The component to preview.
   * @param string $key
   *   The component key.
   *
   * @return array|null
   *   The preview form element or NULL if no demos are available.
   */
  protected function buildComponentPreview(Component $component, string $key): ?array {
    $demos = $this->getSdcDemosForComponent($component);
    if (empty($demos)) {
      return NULL;
    }

    $preview = [
      '#type' => 'fieldset',
      '#title' => $this->t('Preview'),
      '#weight' => -100,
      '#attributes' => [
        'class' => [
          'css-variables-customizer-preview',
        ],
      ],
    ];

    $iframe_id = Html::getId('preview-iframe-' . $key);

    // Build demo options for select list.
    $demo_options = [];
    foreach ($demos as $demo_key => $demo_data) {
      $demo_options[$demo_key] = $demo_data['name'] ?? $demo_key;
    }

    $preview['demo'] = [
      '#type' => 'select',
      '#title' => $this->t('Select Demo'),
      '#options' => $demo_options,
    ];

    if (count($demo_options) > 1) {
      $preview['demo']['#empty_option'] = $this->t('- Select a demo -');
    }

    $preview['show'] = [
      '#type' => 'submit',
      '#value' => $this->t('Show Preview'),
      '#limit_validation_errors' => [],
      '#submit' => [
        [self::class, 'previewSubmit'],
      ],
      '#name' => $component->getPluginId() . ':preview',
      '#ajax' => [
        'callback' => [self::class, 'previewIframeAjaxCallback'],
        'disable-refocus' => TRUE,
        'wrapper' => $iframe_id,
      ],
      '#attributes' => [
        'data-component-id' => $key,
      ],
    ];

    $preview['iframe_container'] = [
      '#type' => 'container',
      '#attributes' => [
        'id' => $iframe_id,
      ],
    ];

    return $preview;
  }

  /**
   * Submit callback to preview components.
   */
  public static function previewSubmit(&$form, FormStateInterface $form_state) {
    $form_state->setRebuild();
  }

  /**
   * Allows overriding a variable for a specific category and group.
   */
  protected function addVariableFormElement(&$form_element, $category, $selector, $variable, $value, $existing_overrides, FormStateInterface $form_state) {
    if (!$form_state->has(['num_overrides', $category, $variable])) {
      $form_state->set(['num_overrides', $category, $variable], !empty($existing_overrides) ? count($existing_overrides) : 1);
    }

    $id = 'css-variables-customizer-customizations-' . $category . '-' . $variable;

    $form_element[$category][$variable] = [
      '#type' => 'fieldset',
      '#prefix' => '<div id="' . $id . '">',
      '#suffix' => '</div>',
      '#title' => new FormattableMarkup(
        ':name (:variable)',
        [
          ':name' => $this->prettifyVariable($variable),
          ':variable' => $variable,
        ]),
    ];

    $num_overrides = $form_state->get(['num_overrides', $category, $variable]);
    for ($i = 0; $i < $num_overrides; $i++) {
      $existing_override = $existing_overrides[$i] ?? [];
      $form_element[$category][$variable][$i] = [
        '#type' => 'container',
        '#attributes' => ['class' => ['container-inline']],
      ];

      $default_selector = $i == 0 ? $value['selector'] ?? $selector : '';
      $form_element[$category][$variable][$i]['selector'] = [
        '#type' => 'textfield',
        '#title' => $this->t('Selector'),
        '#size' => 20,
        '#default_value' => $existing_override['selector'] ?? $default_selector,
        '#placeholder' => $default_selector,
      ];

      $form_element[$category][$variable][$i]['value'] = [
        '#type' => 'textfield',
        '#title' => $this->t('Value'),
        '#attributes' => [
          'class' => [
            'css-variables-customizer-override-value',
          ],
        ],
        '#size' => 20,
        '#default_value' => $existing_override['value'] ?? '',
        '#placeholder' => $i == 0 ? trim($value['value']) : '',
      ];

      if ($i == $num_overrides - 1) {
        $form_element[$category][$variable][$i]['add_more'] = [
          '#type' => 'submit',
          '#value' => $this->t('Add more'),
          '#validate' => [],
          '#name' => $variable . '_add_more',
          '#limit_validation_errors' => [],
          '#ajax' => [
            'callback' => [self::class, 'addMoreVariableVariantsAjaxCallback'],
            'wrapper' => $id,
          ],
          '#submit' => [[self::class, 'addMoreVariableVariantsSubmit']],
        ];
      }
    }
  }

  /**
   * Submit callback to increment the number.
   */
  public static function addMoreVariableVariantsSubmit(&$form, FormStateInterface $form_state) {
    $triggering_element = $form_state->getTriggeringElement();
    $parents = $triggering_element['#parents'];
    $category = $parents[2] ?? '';
    $variable = $parents[3] ?? '';

    $num_overrides = $form_state->get(['num_overrides', $category, $variable]) ?? 1;
    $form_state->set(['num_overrides', $category, $variable], $num_overrides + 1);
    $form_state->setRebuild();
  }

  /**
   * Ajax callback to replace the variables form element.
   */
  public static function addMoreVariableVariantsAjaxCallback(&$form, FormStateInterface $form_state) {
    $triggering_element = $form_state->getTriggeringElement();

    $parents = $triggering_element['#parents'];
    $variable_parents = array_slice($parents, 0, -2);

    $element = NestedArray::getValue($form, $variable_parents);

    return $element ?: ['#markup' => ''];
  }

  /**
   * Ajax callback to show iframe preview.
   */
  public static function previewIframeAjaxCallback(&$form, FormStateInterface $form_state) {
    $triggering_element = $form_state->getTriggeringElement();
    $component_id = $triggering_element['#attributes']['data-component-id'] ?? '';

    // Get the selected demo from form values.
    $parents = array_slice($triggering_element['#parents'], 0, -1);
    $preview_group = NestedArray::getValue($form_state->getValues(), $parents);
    $selected_demo = $preview_group['demo'] ?? '';

    $render_array = [
      '#type' => 'container',
      '#attributes' => [
        'id' => Html::getId('preview-iframe-' . $component_id),
      ],
    ];

    // Only show iframe if a demo is selected.
    if (!empty($selected_demo)) {
      // Get the form object to access the trait methods.
      $form_object = $form_state->getFormObject();

      // Get the component from the component plugin manager.
      $component = $form_object->componentPluginManager->find($component_id);

      if ($component && $form_object instanceof CustomizerForm) {
        // Get the demo URL using the trait method.
        $demo_url = $form_object->getSdcDemoUrl($component, $selected_demo);

        $render_array['iframe'] = [
          '#type' => 'html_tag',
          '#tag' => 'iframe',
          '#attributes' => [
            'src' => $demo_url->toString(),
            'width' => '100%',
            'height' => '600',
            'frameborder' => '0',
            'style' => 'border: 1px solid #ccc; margin-top: 10px;',
          ],
        ];
        return $render_array;
      }
    }

    $render_array['message'] = [
      '#markup' => '<p>' . t('Unable to preview demo.') . '</p>',
    ];

    return $render_array;
  }

  /**
   * Validate the form.
   *
   * When the form is validated, we need to merge the custom variables
   * with the existing variables.
   *
   * @param array $form
   *   The form.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   */
  public function validate(array $form, FormStateInterface $form_state) {

    $customizations = $form_state->getValue('customizations');
    $customizations_config = [];
    if ($this->validateCustom($form, $customizations, $form_state)) {
      $custom_variables = $this->cssVariablesManager->getStringVariables($customizations['custom']['variables'] ?? []);
      $form_state->setValue('custom', $custom_variables);
    }
    else {
      $custom_variables = [];
    }

    foreach ($customizations as $key => $key_customizations) {
      $all_variables = [];
      if (!is_array($key_customizations) || $key == 'custom') {
        continue;
      }

      $variables = [];
      foreach ($key_customizations['categories'] ?? [] as $group => $category_variables) {
        $all_variables = array_merge($all_variables, array_keys($category_variables));
        $variables = array_merge($variables, $this->massageCategoryVariables($category_variables, $form['customizations'][$key][$group], $form_state));
      }

      $common_variables = array_intersect($all_variables, array_keys($custom_variables));

      if (empty($common_variables) && (!empty($variables) || !empty($custom_variables))) {
        $customizations_config[] = [
          'id' => $key,
          'overrides' => $variables,
        ];
      }
      elseif (!empty($common_variables)) {
        $form_state->setError(
          $form['customizations']['custom']['variables'],
          t('The following variables are defined in the @group group: @variables', [
            '@group' => $key,
            '@variables' => implode(', ', $common_variables),
          ]));
      }
    }

    if (!$form_state->getErrors()) {
      $form_state->setValue('customizations', $customizations_config);

      // Store preview data in tempstore only for AJAX callbacks.
      $theme = $form_state->get('theme');
      if ($theme && $form_state->isProcessingInput() && $form_state->getTriggeringElement() && !empty($form_state->getTriggeringElement()['#ajax'])) {
        $preview_data = [
          'customizations' => $customizations_config,
          'custom' => $custom_variables,
        ];
        $this->tempStore->set('preview_' . $theme, $preview_data);
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $theme = $form_state->get('theme');

    $this->config($this->getThemeConfigName($theme))
      ->set('customizations', $form_state->getValue('customizations'))
      ->set('custom', $form_state->getValue('custom'))
      ->save();

    parent::submitForm($form, $form_state);
  }

  /**
   * Massage category variables.
   *
   * It receives the list of variables for a specific group and
   * returns all values cleaned up:
   *   - No empty values.
   *   - No specific UI elements (add-more buttons).
   */
  protected function massageCategoryVariables(array $category_variables, &$form_element, FormStateInterface $form_state) {
    $variables = [];
    foreach ($category_variables as $name => $variants) {
      if (!is_array($variants)) {
        continue;
      }
      $valid_variables = [];
      $used_selectors = [];
      foreach ($variants as $key => $variant) {
        if (!empty($variant['value']) && !empty($variant['selector']) && !in_array($variant['selector'], $used_selectors)) {
          $valid_variables[] = [
            'value' => $variant['value'],
            'selector' => $variant['selector'],
          ];
          $used_selectors[] = $variant['selector'];
        }
        elseif (in_array($variant['selector'], $used_selectors)) {
          $form_state->setError($form_element[$name][$key], t('Repeated selector for @variable variable: @selector', [
            '@variable' => $name,
            '@selector' => $variant['selector'],
          ]));
        }
      }
      $variables[$name] = $valid_variables;
    }
    return array_filter($variables);
  }

  /**
   * Validate each customization has the CSS variable pattern.
   *
   * Valid variable example:
   * --theme-primary: #000000;
   *
   * @param array $form
   *   The form.
   * @param array $customizations
   *   The customizations.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   */
  protected function validateCustom(array $form, array $customizations, FormStateInterface $form_state) {
    $custom = $customizations['custom']['variables'] ?? '';
    $custom_variables = array_filter(explode(PHP_EOL, $custom));
    $invalid_variables = [];
    foreach ($custom_variables as $custom_variable) {
      if (!preg_match(CssVariablesManagerInterface::CSS_VARIABLES_REGEXP, $custom_variable)) {
        $invalid_variables[] = $custom_variable;
      }
    }

    if (!empty($invalid_variables)) {
      $form_state->setError($form['customizations']['custom']['variables'], t('Invalid CSS variables: @variables', ['@variables' => implode(', ', $invalid_variables)]));
      return FALSE;
    }
    return TRUE;
  }

  /**
   * Get the customization for a given key.
   *
   * @param string $key
   *   The key of the customization.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   Used to get the theme name.
   *
   * @return array
   *   The customization.
   */
  protected function getCustomizationsForKey($key, FormStateInterface $form_state) {
    $theme = $form_state->get('theme');
    if (!empty($theme)) {
      $config = $this->config($this->getThemeConfigName($theme));
      $customizations = $config->get('customizations') ?? [];
      foreach ($customizations as $customization) {
        if ($customization['id'] == $key) {
          return $customization;
        }
      }
    }

    return [];
  }

  /**
   * Prettify a variable name.
   *
   * @param string $variable
   *   The variable name.
   *
   * @return string
   *   The prettified variable name.
   */
  protected function prettifyVariable(string $variable) {
    return ucfirst(str_replace(['--', '-'], ['', ' '], $variable));
  }

  /**
   * Get the css assets from current theme.
   *
   * @return array
   *   List of files.
   */
  protected function getMainCssAssets(Extension $theme) {
    $assets = [];
    $path = dirname($theme->getPathname());
    $stylesheets = $theme->info['css_variables_customizer']['stylesheets'] ?? [];
    foreach ($stylesheets as $stylesheet) {
      $stylesheet_path = $path . '/' . $stylesheet;
      if (is_dir($stylesheet_path)) {
        $assets = array_merge($assets, glob($stylesheet_path . '/[!_]*.css'));
        $assets = array_merge($assets, glob($stylesheet_path . '/[!_]*.scss'));
      }
      elseif (file_exists($stylesheet_path)) {
        $assets[] = $stylesheet_path;
      }
    }
    return $assets;
  }

}
