/**
 * @file
 * Nuxt preview provider behavior.
 *
 * Integrates with the NuxtPreviewProvider PHP plugin to render
 * custom elements using the Nuxt component-preview module.
 */

((Drupal, drupalSettings, once) => {
  Drupal.behaviors.customElementsNuxtPreview = {
    attach(context, settings) {
      if (!settings.customElementsNuxtPreview?.baseUrl) {
        console.error('Nuxt preview baseUrl not configured');
        return;
      }

      // Load the Nuxt app loader script only once.
      once('nuxt-preview-app-loader', 'head', context).forEach(() => {
        const script = document.createElement('script');
        script.src = `${settings.customElementsNuxtPreview.baseUrl}/nuxt-component-preview/app-loader.js`;
        document.head.appendChild(script);
      });

      // Helper function that handles both sync and async cases.
      const onNuxtComponentPreviewReady = (callback) =>
        window.__nuxtComponentPreviewApp
          ? callback(window.__nuxtComponentPreviewApp)
          : window.addEventListener(
              'nuxt-component-preview:ready',
              (event) => callback(event.detail.nuxtApp),
              { once: true },
            );

      // Initialize containers. Nested components will be handled by
      // re-attaching behaviors after rendering.
      once('nuxt-preview', '.nuxt-preview-container', context).forEach(
        (container) => {
          onNuxtComponentPreviewReady(async (nuxtApp) => {
            await nuxtApp.$previewComponent(
              container.dataset.componentName,
              JSON.parse(container.dataset.componentProps || '{}'),
              JSON.parse(container.dataset.componentSlots || '{}'),
              `#${container.id}`,
            );

            // Wait for DOM updates and innerHTML parsing. Add 50ms buffer to make sure things are updated in the DOM.
            await new Promise((resolve) => {
              requestAnimationFrame(() =>
                requestAnimationFrame(() => resolve()),
              );
            });
            await new Promise((resolve) => {
              setTimeout(() => resolve(), 50);
            });

            // Re-attach Drupal behaviors to process nested containers.
            Drupal.attachBehaviors(context);
          });
        },
      );
    },
  };
})(Drupal, drupalSettings, once);
