<?php

namespace Drupal\custom_elements\Plugin\CustomElementsPreviewProvider;

use drunomics\ServiceUtils\Core\Render\RendererTrait;
use Drupal\Component\Utility\Html;
use Drupal\Core\Render\BubbleableMetadata;
use Drupal\custom_elements\CustomElement;
use Drupal\custom_elements\PreviewProvider\CustomElementsPreviewProviderBase;

/**
 * Provides a preview using a Nuxt frontend.
 *
 * @CustomElementsPreviewProvider(
 *   id = "nuxt",
 *   label = @Translation("JavaScript - Nuxt"),
 *   description = @Translation("Generates a preview using the Nuxt component-preview module")
 * )
 */
class NuxtPreviewProvider extends CustomElementsPreviewProviderBase {

  use RendererTrait;

  /**
   * {@inheritdoc}
   */
  public function preview(CustomElement $element): array {
    // The base URL should always be set when this method is called,
    // as isApplicable() checks for it.
    assert(!empty($this->baseUrl), 'Base URL must be set when preview() is called');

    // Generate a unique element ID for this preview container.
    $element_id = Html::getUniqueId('nuxt-preview-' . $element->getTag());

    // Convert the custom element to JSON array representation.
    $cache_metadata = new BubbleableMetadata();
    $cache_metadata->addCacheableDependency($element);

    // Prepare slots.
    $slots = [];
    foreach ($element->getSortedSlots() as $slot_entry) {
      if ($slot_entry['content'] instanceof CustomElement) {
        $slots[$slot_entry['key']][] = $this->preview($slot_entry['content']);
        $cache_metadata->addCacheableDependency($slot_entry['content']);
      }
      else {
        $slots[$slot_entry['key']][] = [
          '#type' => 'markup',
          '#markup' => $slot_entry['content'],
        ];
      }
    }
    // Render each slot into markup.
    $slot_markup = [];
    foreach ($slots as $key => $slot) {
      $slot_markup[$key] = $this->getrenderer()->renderInIsolation($slot);
    }

    // Build the render array for the preview.
    $build = [
      '#type' => 'html_tag',
      '#tag' => 'div',
      '#attributes' => [
        'id' => $element_id,
        'class' => ['nuxt-preview-container'],
        'data-component-name' => $element->getTag(),
        'data-component-props' => json_encode($element->getAttributes()),
        'data-component-slots' => json_encode($slot_markup),
      ],
    ];

    // Apply cache metadata from the element.
    $cache_metadata->applyTo($build);

    // Add attachments after cache metadata to prevent overwriting.
    $build['#attached']['library'][] = 'custom_elements/nuxt_preview';
    $build['#attached']['drupalSettings']['customElementsNuxtPreview']['baseUrl'] = $this->baseUrl;
    return $build;
  }

}
