/**
 * @file
 * Attaches the behaviors for the Custom Elements UI module.
 *
 * Stripped down and modified copy of Core field_ui.js.
 */

(function ($, Drupal, drupalSettings, debounce) {
  /**
   * Attaches the customElementsUIOverview behavior.
   *
   * @type {Drupal~behavior}
   *
   * @prop {Drupal~behaviorAttach} attach
   *   Attaches the customElementsUIOverview behavior.
   *
   * @see Drupal.customElementsUIOverview.attach
   */
  Drupal.behaviors.customElementsUIDisplayOverview = {
    attach(context, settings) {
      once(
        'field-display-overview',
        'table#field-display-overview',
        context,
      ).forEach((overview) => {
        Drupal.customElementsUIOverview.attach(
          overview,
          settings.fieldUIRowsData,
          Drupal.customElementsUIDisplayOverview,
        );
      });
    },
  };

  // Override the beforeSend method to disable the submit button until
  // the AJAX request is completed. This is done to avoid the race
  // condition that is being caused by change event listener that is
  // attached to every form element inside field storage config edit
  // form to update the field config form based on changes made to the
  // storage settings.
  const originalAjaxBeforeSend = Drupal.Ajax.prototype.beforeSend;
  // eslint-disable-next-line func-names
  Drupal.Ajax.prototype.beforeSend = function () {
    // Disable the submit button on AJAX request initiation.
    $('.field-config-edit-form [data-drupal-selector="edit-submit"]').prop(
      'disabled',
      true,
    );
    // eslint-disable-next-line prefer-rest-params
    return originalAjaxBeforeSend.apply(this, arguments);
  };
  // Re-enable the submit button after AJAX request is completed.
  // eslint-disable-next-line
  $(document).on('ajaxComplete', () => {
    $('.field-config-edit-form [data-drupal-selector="edit-submit"]').prop(
      'disabled',
      false,
    );
  });

  /**
   * Namespace for the field UI overview.
   *
   * @namespace
   */
  Drupal.customElementsUIOverview = {
    /**
     * Attaches the customElementsUIOverview behavior.
     *
     * @param {HTMLTableElement} table
     *   The table element for the overview.
     * @param {object} rowsData
     *   The data of the rows in the table.
     * @param {object} rowHandlers
     *   Handlers to be added to the rows.
     */
    attach(table, rowsData, rowHandlers) {
      const tableDrag = Drupal.tableDrag[table.id];

      // Add custom tabledrag callbacks.
      tableDrag.onDrop = this.onDrop;
      tableDrag.row.prototype.onSwap = this.onSwap;

      // Create row handlers.
      $(table)
        .find('tr.draggable')
        .each(function () {
          // Extract server-side data for the row.
          const row = this;
          if (row.id in rowsData) {
            const data = rowsData[row.id];
            data.tableDrag = tableDrag;

            // Create the row handler, make it accessible from the DOM row
            // element.
            const rowHandler = new rowHandlers[data.rowHandler](row, data);
            $(row).data('customElementsUIRowHandler', rowHandler);
          }
        });
    },

    /**
     * Event handler to be attached to form inputs triggering a region change.
     */
    onChange() {
      const $trigger = $(this);
      const $row = $trigger.closest('tr');

      // Do not fire change listeners for items within forms that have their
      // own AJAX callbacks to process a change.
      if ($trigger.closest('.ajax-new-content').length !== 0) {
        return;
      }

      const rowHandler = $row.data('customElementsUIRowHandler');
      const refreshRows = {};
      refreshRows[rowHandler.name] = $trigger.get(0);

      // Region change handling removed - we don't use regions for organization.

      // Fields inside `.tabledrag-hide` are typically hidden. They can be
      // visible when "Show row weights" are enabled. If their value is changed
      // while visible, the row should be marked as changed, but they should not
      // be processed via AJAXRefreshRows as they are intended to be fields AJAX
      // updates the value of.
      if ($trigger.closest('.tabledrag-hide').length) {
        const thisTableDrag = Drupal.tableDrag['field-display-overview'];
        // eslint-disable-next-line new-cap
        const rowObject = new thisTableDrag.row(
          $row[0],
          '',
          thisTableDrag.indentEnabled,
          thisTableDrag.maxDepth,
          true,
        );
        rowObject.markChanged();
        rowObject.addChangedWarning();
      } else {
        // Ajax-update the rows.
        Drupal.customElementsUIOverview.AJAXRefreshRows(refreshRows);
      }
    },

    /**
     * Lets row handlers react when a row is dropped into a new region.
     */
    onDrop() {
      const dragObject = this;
      const row = dragObject.rowObject.element;
      const $row = $(row);
      const rowHandler = $row.data('customElementsUIRowHandler');
      if (typeof rowHandler !== 'undefined') {
        const regionRow = $row.prevAll('tr.region-message').get(0);
        const region = regionRow.className.replace(
          /([^ ]+[ ]+)*region-([^ ]+)-message([ ]+[^ ]+)*/,
          '$2',
        );

        if (region !== rowHandler.region) {
          // Let the row handler deal with the region change.
          const refreshRows = rowHandler.regionChange(region);
          // Update the row region.
          rowHandler.region = region;
          // Ajax-update the rows.
          Drupal.customElementsUIOverview.AJAXRefreshRows(refreshRows);
        }
      }
    },

    /**
     * Refreshes placeholder rows in empty regions while a row is being dragged.
     *
     * Copied from block.js.
     *
     * @param {HTMLElement} draggedRow
     *   The tableDrag rowObject for the row being dragged.
     */
    onSwap(draggedRow) {
      const rowObject = this;
      $(rowObject.table)
        .find('tr.region-message')
        .each(function () {
          const $this = $(this);
          // If the dragged row is in this region, but above the message row, swap
          // it down one space.
          if (
            $this.prev('tr').get(0) ===
            rowObject.group[rowObject.group.length - 1]
          ) {
            // Prevent a recursion problem when using the keyboard to move rows
            // up.
            if (
              rowObject.method !== 'keyboard' ||
              rowObject.direction === 'down'
            ) {
              rowObject.swap('after', this);
            }
          }
          // This region has become empty.
          if (
            $this.next('tr').length === 0 ||
            !$this.next('tr')[0].matches('.draggable')
          ) {
            $this.removeClass('region-populated').addClass('region-empty');
          }
          // This region has become populated.
          else if (this.matches('.region-empty')) {
            $this.removeClass('region-empty').addClass('region-populated');
          }
        });
    },

    /**
     * Triggers Ajax refresh of selected rows.
     *
     * The 'format type' selects can trigger a series of changes in child rows.
     * The #ajax behavior is therefore not attached directly to the selects, but
     * triggered manually through a hidden #ajax 'Refresh' button.
     *
     * @param {object} rows
     *   A hash object, whose keys are the names of the rows to refresh (they
     *   will receive the 'ajax-new-content' effect on the server side), and
     *   whose values are the DOM element in the row that should get an Ajax
     *   throbber.
     */
    AJAXRefreshRows(rows) {
      // Separate keys and values.
      const rowNames = [];
      const ajaxElements = [];
      Object.keys(rows || {}).forEach((rowName) => {
        rowNames.push(rowName);
        ajaxElements.push(rows[rowName]);
      });
      if (rowNames.length) {
        // Add a throbber next each of the ajaxElements.
        $(ajaxElements).after(Drupal.theme.ajaxProgressThrobber());
        const $refreshRows = $('input[name=refresh_rows]');
        if ($refreshRows.length) {
          // Fire the Ajax update.
          $refreshRows[0].value = rowNames.join(' ');
        }
        once(
          'edit-refresh',
          'input[data-drupal-selector="edit-refresh"]',
        ).forEach((input) => {
          // Keep track of the element that was focused prior to triggering the
          // mousedown event on the hidden submit button.
          let returnFocus = {
            drupalSelector: null,
            scrollY: null,
          };
          // Use jQuery on to listen as the mousedown event is propagated by
          // jQuery trigger().
          $(input).on('mousedown', () => {
            returnFocus = {
              drupalSelector: document.activeElement.hasAttribute(
                'data-drupal-selector',
              )
                ? document.activeElement.getAttribute('data-drupal-selector')
                : false,
              scrollY: window.scrollY,
            };
          });
          input.addEventListener('focus', () => {
            if (returnFocus.drupalSelector) {
              // Refocus the element that lost focus due to this hidden submit
              // button being triggered by a mousedown event.
              document
                .querySelector(
                  `[data-drupal-selector="${returnFocus.drupalSelector}"]`,
                )
                .focus();
            }
            // Ensure the scroll position is the same as when the input was
            // initially changed.
            window.scrollTo({
              top: returnFocus.scrollY,
            });
            returnFocus = {};
          });
        });
        $('input[data-drupal-selector="edit-refresh"]').trigger('mousedown');

        // Disabled elements do not appear in POST ajax data, so we mark the
        // elements disabled only after firing the request.
        $(ajaxElements).prop('disabled', true);
      }
    },
  };

  /**
   * Row handlers for the 'Manage display' screen.
   *
   * @namespace
   */
  Drupal.customElementsUIDisplayOverview = {};

  /**
   * Constructor for a 'field' row handler.
   *
   * This handler is used for both fields and 'extra fields' rows.
   *
   * @constructor
   *
   * @param {HTMLTableRowElement} row
   *   The row DOM element.
   * @param {object} data
   *   Additional data to be populated in the constructed object.
   *
   * @return {Drupal.customElementsUIDisplayOverview.field}
   *   The field row handler constructed.
   */
  Drupal.customElementsUIDisplayOverview.field = function (row, data) {
    this.row = row;
    this.name = data.name;
    this.region = data.region;
    this.tableDrag = data.tableDrag;
    this.defaultPlugin = data.defaultPlugin;
    this.$pluginSelect = $(row).find('.field-plugin-type');
    // Region select not used - all rows in 'content' region.
    this.$regionSelect = [];

    // Attach change listeners to select and input elements in the row.
    $(row)
      .find('select')
      .on('change', Drupal.customElementsUIOverview.onChange);

    return this;
  };

  Drupal.customElementsUIDisplayOverview.field.prototype = {
    /**
     * Returns the region corresponding to the current form values of the row.
     *
     * @return {string}
     *   Always 'content' since we don't use regions for organization.
     */
    getRegion() {
      // Region concept not used - always return 'content'.
      return 'content';
    },

    /**
     * Reacts to a row being changed regions.
     *
     * This function is called when the row is moved to a different region, as
     * a
     * result of either :
     * - a drag-and-drop action (the row's form elements then probably need to
     * be updated accordingly)
     * - user input in one of the form elements watched by the
     *   {@link Drupal.customElementsUIOverview.onChange} change listener.
     *
     * @param {string} region
     *   The name of the new region for the row.
     *
     * @return {object}
     *   A hash object indicating which rows should be Ajax-updated as a result
     *   of the change, in the format expected by
     *   {@link Drupal.customElementsUIOverview.AJAXRefreshRows}.
     */
    regionChange(region) {
      // Region concept not used - this is a no-op.
      // All rows stay in 'content' region.
      return {};
    },
  };
})(jQuery, Drupal, drupalSettings, Drupal.debounce);
