<?php

declare(strict_types=1);

namespace Drupal\custom_field\Plugin\Field\FieldWidget;

use Drupal\Component\Plugin\Exception\PluginException;
use Drupal\Component\Utility\NestedArray;
use Drupal\Core\Ajax\AjaxResponse;
use Drupal\Core\Ajax\ReplaceCommand;
use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Field\FieldStorageDefinitionInterface;
use Drupal\Core\Field\WidgetBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\custom_field\Plugin\CustomFieldTypeInterface;
use Drupal\custom_field\Plugin\CustomFieldTypeManagerInterface;
use Drupal\custom_field\Plugin\CustomFieldWidgetManagerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\Validator\ConstraintViolationInterface;
use Symfony\Component\Validator\ConstraintViolationListInterface;

/**
 * Base widget definition for custom field type.
 */
abstract class CustomWidgetBase extends WidgetBase {

  /**
   * {@inheritdoc}
   */
  public static function defaultSettings(): array {
    return [
      'label' => TRUE,
      'wrapper' => 'div',
      'open' => TRUE,
      'fields' => [],
    ] + parent::defaultSettings();
  }

  /**
   * Constructs a custom field widget.
   *
   * @param string $plugin_id
   *   The plugin ID for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Field\FieldDefinitionInterface $field_definition
   *   The definition of the field to which the widget is associated.
   * @param array $settings
   *   The widget settings.
   * @param array $third_party_settings
   *   Any third party settings.
   * @param \Drupal\custom_field\Plugin\CustomFieldTypeManagerInterface $customFieldTypeManager
   *   The custom field type manager.
   * @param \Drupal\custom_field\Plugin\CustomFieldWidgetManagerInterface $customFieldWidgetManager
   *   The custom field widget manager.
   */
  public function __construct($plugin_id, $plugin_definition, FieldDefinitionInterface $field_definition, array $settings, array $third_party_settings, protected CustomFieldTypeManagerInterface $customFieldTypeManager, protected CustomFieldWidgetManagerInterface $customFieldWidgetManager) {
    parent::__construct($plugin_id, $plugin_definition, $field_definition, $settings, $third_party_settings);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $plugin_id,
      $plugin_definition,
      $configuration['field_definition'],
      $configuration['settings'],
      $configuration['third_party_settings'],
      $container->get('plugin.manager.custom_field_type'),
      $container->get('plugin.manager.custom_field_widget')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function settingsForm(array $form, FormStateInterface $form_state): array {
    $definition = $this->fieldDefinition;
    $field_name = $definition->getName();
    $field_settings = $this->getSetting('fields') ?? [];
    $custom_items = $this->sortFields($field_settings);
    $values = $form_state->getValues();

    $elements = parent::settingsForm($form, $form_state);
    $elements['#tree'] = TRUE;
    $elements['#attached']['library'][] = 'custom_field/customfield-admin';

    $elements['label'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Show field label?'),
      '#default_value' => $this->getSetting('label'),
    ];
    $elements['wrapper'] = [
      '#type' => 'select',
      '#title' => $this->t('Wrapper'),
      '#default_value' => $this->getSetting('wrapper'),
      '#options' => [
        'div' => $this->t('Default'),
        'fieldset' => $this->t('Fieldset'),
        'details' => $this->t('Details'),
      ],
      '#states' => [
        'visible' => [
          'input[name="fields[' . $field_name . '][settings_edit_form][settings][label]"]' => ['checked' => TRUE],
        ],
      ],
    ];
    $elements['open'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Show open by default?'),
      '#default_value' => $this->getSetting('open'),
      '#states' => [
        'visible' => [
          'input[name="fields[' . $field_name . '][settings_edit_form][settings][label]"]' => ['checked' => TRUE],
          0 => 'AND',
          'select[name="fields[' . $field_name . '][settings_edit_form][settings][wrapper]"]' => ['value' => 'details'],
        ],
      ],
    ];
    $elements['fields'] = [
      '#type' => 'table',
      '#header' => [
        $this->t('Field settings'),
        $this->t('Weight'),
      ],
      '#tableselect' => FALSE,
      '#tabledrag' => [
        [
          'action' => 'order',
          'relationship' => 'sibling',
          'group' => 'field-settings-order-weight',
        ],
      ],
      '#weight' => 10,
      '#attributes' => [
        'class' => ['form-fields-settings-table'],
      ],
    ];

    foreach ($custom_items as $name => $custom_item) {
      $plugin_id = $custom_item->getPluginId();
      $value_keys = [
        'fields',
        $field_name,
        'settings_edit_form',
        'settings',
        'fields',
        $name,
      ];
      $wrapper_id = 'field-' . $field_name . '-' . $name;

      // UUid fields have no configuration.
      if ($plugin_id === 'uuid') {
        continue;
      }
      $settings = $field_settings[$name] ?? [];
      $weight = $settings['weight'] ?? 0;
      $options = self::getCustomFieldWidgetOptions($custom_item);
      $options_count = count($options);
      $widget_type = $settings['type'] ?? NULL;

      if (!empty($widget_type) && in_array($widget_type, $this->customFieldWidgetManager->getWidgetsForField($plugin_id))) {
        $type = $widget_type;
      }
      else {
        $type = $custom_item->getDefaultWidget();
      }
      if (!empty($values)) {
        $type = NestedArray::getValue($values, [...$value_keys, 'type']) ?? $type;
      }
      $open = FALSE;
      // Keep details open when type changes.
      if ($form_state->isRebuilding()) {
        $trigger = $form_state->getTriggeringElement();
        if (in_array($name, $trigger['#parents']) && end($trigger['#parents']) === 'type') {
          $open = TRUE;
        }
      }

      $elements['fields'][$name] = [
        '#attributes' => [
          'class' => ['draggable'],
        ],
        '#weight' => $weight,
      ];
      $elements['fields'][$name]['settings'] = [
        '#type' => 'details',
        '#title' => $this->t('@label', ['@label' => $custom_item->getLabel()]),
        '#parents' => $value_keys,
        '#open' => $open,
        '#prefix' => '<div id="' . $wrapper_id . '">',
        '#suffix' => '</div>',
      ];
      $elements['fields'][$name]['settings']['type'] = [
        '#type' => 'select',
        '#title' => $this->t('%name widget', ['%name' => $name]),
        '#options' => $options,
        '#default_value' => $type,
        '#value' => $type,
        '#ajax' => [
          'callback' => [$this, 'widgetTypeCallback'],
          'wrapper' => $wrapper_id,
        ],
        '#attributes' => [
          'disabled' => $options_count <= 1,
        ],
      ];

      $plugin_options = $this->customFieldWidgetManager->createOptionsForInstance(
        $field_name,
        $custom_item,
        $type,
        $settings,
        'default'
      );
      /** @var \Drupal\custom_field\Plugin\CustomFieldWidgetInterface $widget */
      $widget = $this->customFieldWidgetManager->getInstance($plugin_options);
      $elements['fields'][$name]['settings'] += $widget->widgetSettingsForm($form_state, $custom_item);
      $elements['fields'][$name]['weight'] = [
        '#type' => 'weight',
        '#title' => $this->t('Weight for @label', ['@label' => $custom_item->getLabel()]),
        '#title_display' => 'invisible',
        '#default_value' => $weight,
        '#attributes' => ['class' => ['field-settings-order-weight']],
      ];
    }

    return $elements;
  }

  /**
   * Ajax callback for changing widget type.
   *
   * Selects and returns the fieldset with the names in it.
   *
   * @param array<string, mixed> $form
   *   The form.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   *
   * @return \Drupal\Core\Ajax\AjaxResponse
   *   The updated form element.
   */
  public function widgetTypeCallback(array $form, FormStateInterface $form_state): AjaxResponse {
    $trigger = $form_state->getTriggeringElement();
    $wrapper_id = $trigger['#ajax']['wrapper'];

    // Get the current parent array for this widget.
    $parents = $trigger['#array_parents'];
    $sliced_parents = array_slice($parents, 0, -1, TRUE);

    // Get the updated element from the form structure.
    $updated_element = NestedArray::getValue($form, $sliced_parents);

    // Create an AjaxResponse.
    $response = new AjaxResponse();
    $response->addCommand(new ReplaceCommand('#' . $wrapper_id, $updated_element));

    return $response;
  }

  /**
   * {@inheritdoc}
   */
  public function settingsSummary(): array {
    $summary = [];

    $summary[] = $this->t('Show field label?: @label', ['@label' => $this->getSetting('label') ? 'Yes' : 'No']);
    $summary[] = $this->t('Wrapper: @wrapper', ['@wrapper' => $this->getSetting('wrapper')]);
    if ($this->getSetting('wrapper') === 'details') {
      $summary[] = $this->t('Open: @open', ['@open' => $this->getSetting('open') ? 'Yes' : 'No']);
    }

    return $summary;
  }

  /**
   * {@inheritdoc}
   */
  public function formElement(FieldItemListInterface $items, $delta, array $element, array &$form, FormStateInterface $form_state): array {
    $element['#attached']['library'][] = 'custom_field/custom-field-widget';
    $element['#attributes']['class'][] = 'custom-field-widget-wrapper';
    $element['#type'] = 'container';
    if ($this->getSetting('label')) {
      $wrapper = $this->getSetting('wrapper');
      if ($wrapper === 'fieldset') {
        $element['#type'] = 'fieldset';
      }
      elseif ($wrapper === 'details') {
        $element['#type'] = 'details';
        $element['#open'] = $this->getSetting('open');
      }
    }

    return $element;
  }

  /**
   * Get the field storage definition.
   *
   * @return \Drupal\Core\Field\FieldStorageDefinitionInterface
   *   The field storage definition.
   */
  public function getFieldStorageDefinition(): FieldStorageDefinitionInterface {
    return $this->fieldDefinition->getFieldStorageDefinition();
  }

  /**
   * Get the custom field items for this field.
   *
   * @return \Drupal\custom_field\Plugin\CustomFieldTypeInterface[]
   *   An array of custom field items.
   */
  public function getCustomFieldItems(): array {
    return $this->customFieldTypeManager->getCustomFieldItems($this->fieldDefinition->getSettings());
  }

  /**
   * {@inheritdoc}
   */
  public function massageFormValues(array $values, array $form, FormStateInterface $form_state): array {
    $columns = $this->getFieldSetting('columns');
    $plugins = $this->getWidgetPlugins();
    foreach ($values as &$item) {
      foreach ($item as $field_name => &$field_value) {
        $plugin = $plugins[$field_name] ?? NULL;

        if ($plugin && method_exists($plugin, 'massageFormValue')) {
          $field_value = $plugin->massageFormValue($field_value, $columns[$field_name] ?? '');
        }
      }
    }

    return $values;
  }

  /**
   * {@inheritdoc}
   */
  protected function formMultipleElements(FieldItemListInterface $items, array &$form, FormStateInterface $form_state): array {
    $parents = $form['#parents'];
    $field_name = $this->fieldDefinition->getName();
    $cardinality = $this->fieldDefinition->getFieldStorageDefinition()->getCardinality();
    $processed_flag = "custom_field_{$field_name}_processed";
    if (!empty($parents)) {
      $id_suffix = implode('_', $parents);
      $processed_flag .= "_{$id_suffix}";
    }

    // If we're using unlimited cardinality we don't display one empty item.
    // Form validation will kick in if left empty which essentially means
    // people won't be able to submit without filling required fields for
    // another value.
    if ($cardinality === FieldStorageDefinitionInterface::CARDINALITY_UNLIMITED && count($items) > 0 && !$form_state->get($processed_flag)) {
      $field_state = static::getWidgetState($parents, $field_name, $form_state);
      if (empty($field_state['array_parents'])) {
        --$field_state['items_count'];
        static::setWidgetState($parents, $field_name, $form_state, $field_state);

        // Set a flag on the form denoting that we've already removed the empty
        // item that is usually appended to the end on fresh form loads.
        $form_state->set($processed_flag, TRUE);
      }
    }

    return parent::formMultipleElements($items, $form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function errorElement(array $element, ConstraintViolationInterface $error, array $form, FormStateInterface $form_state) {
    $path = explode('.', $error->getPropertyPath());
    $field_name = (string) end($path);
    $plugins = $this->getWidgetPlugins();
    if (!empty($element[$field_name]) && isset($plugins[$field_name])) {
      $plugin = $plugins[$field_name];
      if (method_exists($plugin, 'errorElement')) {
        return $plugin->errorElement($element, $error, $form, $form_state);
      }
    }
    return isset($error->arrayPropertyPath[0]) ? $element[$error->arrayPropertyPath[0]] : $element;
  }

  /**
   * {@inheritdoc}
   */
  public function calculateDependencies(): array {
    $dependencies = parent::calculateDependencies();
    $plugins = $this->getWidgetPlugins();
    foreach ($plugins as $plugin) {
      $plugin_dependencies = $plugin->calculateWidgetDependencies();
      $dependencies = array_merge_recursive($dependencies, $plugin_dependencies);
    }

    return $dependencies;
  }

  /**
   * {@inheritdoc}
   */
  public function onDependencyRemoval(array $dependencies): bool {
    $changed = parent::onDependencyRemoval($dependencies);
    $plugins = $this->getWidgetPlugins();
    $fields = $this->getSetting('fields');
    foreach ($plugins as $name => $plugin) {
      $changed_settings = $plugin->onWidgetDependencyRemoval($dependencies);
      if (!empty($changed_settings) && isset($fields[$name])) {
        $fields[$name] = $changed_settings;
        $changed = TRUE;
      }
    }
    if ($changed) {
      $this->setSetting('fields', $fields);
    }

    return $changed;
  }

  /**
   * Reports field-level validation errors against actual form elements.
   *
   * @param \Drupal\Core\Field\FieldItemListInterface<\Drupal\custom_field\Plugin\Field\FieldType\CustomItem> $items
   *   The field values.
   * @param \Symfony\Component\Validator\ConstraintViolationListInterface $violations
   *   A list of constraint violations to flag.
   * @param array<string, mixed> $form
   *   The form structure where field elements are attached to. This might be a
   *   full form structure, or a sub-element of a larger form.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   */
  public function flagErrors(FieldItemListInterface $items, ConstraintViolationListInterface $violations, array $form, FormStateInterface $form_state): void {
    $plugins = $this->getWidgetPlugins();
    foreach ($plugins as $plugin) {
      if (method_exists($plugin, 'flagErrors')) {
        $plugin->flagErrors($items, $violations, $form, $form_state);
      }
    }

    parent::flagErrors($items, $violations, $form, $form_state);
  }

  /**
   * Helper function to sort field settings by weight.
   *
   * @param array $settings
   *   The field settings.
   *
   * @return array<string, CustomFieldTypeInterface>
   *   The sorted custom items.
   */
  protected function sortFields(array $settings): array {
    $custom_items = $this->getCustomFieldItems();

    // Sort items by weight.
    uasort($custom_items, function (CustomFieldTypeInterface $a, CustomFieldTypeInterface $b) use ($settings) {
      $weight_a = $settings[$a->getName()]['weight'] ?? 0;
      $weight_b = $settings[$b->getName()]['weight'] ?? 0;
      return $weight_a <=> $weight_b;
    });

    return $custom_items;
  }

  /**
   * Return the available widget plugins as an array keyed by plugin_id.
   *
   * @param \Drupal\custom_field\Plugin\CustomFieldTypeInterface $custom_item
   *   The Custom field type interface.
   *
   * @return array<string, mixed>
   *   The array of widget options.
   *
   * @throws \Drupal\Component\Plugin\Exception\PluginException
   */
  private static function getCustomFieldWidgetOptions(CustomFieldTypeInterface $custom_item): array {
    $options = [];
    /** @var \Drupal\custom_field\Plugin\CustomFieldWidgetManager $plugin_service */
    $plugin_service = \Drupal::service('plugin.manager.custom_field_widget');
    $definitions = $plugin_service->getDefinitions();
    $type = $custom_item->getPluginId();
    // Remove undefined widgets for data_type.
    foreach ($definitions as $key => $definition) {
      /** @var \Drupal\custom_field\Plugin\CustomFieldWidgetInterface $instance */
      $instance = $plugin_service->createInstance($definition['id']);
      if (!$instance::isApplicable($custom_item)) {
        unset($definitions[$key]);
      }
      if (!in_array($type, $definition['field_types'])) {
        unset($definitions[$key]);
      }
    }
    // Sort the widgets by category and then by name.
    uasort($definitions, function ($a, $b) {
      if ($a['category'] != $b['category']) {
        return strnatcasecmp((string) $a['category'], (string) $b['category']);
      }
      return strnatcasecmp((string) $a['label'], (string) $b['label']);
    });
    foreach ($definitions as $id => $definition) {
      $category = $definition['category'];
      // Add category grouping for multiple options.
      $options[(string) $category][$id] = $definition['label'];
    }
    if (count($options) <= 1) {
      $options = array_values($options)[0];
    }

    return $options;
  }

  /**
   * Helper function to fetch field widget plugins.
   *
   * @return array<string, \Drupal\custom_field\Plugin\CustomFieldWidgetInterface>
   *   An array of widget plugins.
   */
  protected function getWidgetPlugins(): array {
    $plugins = [];
    $fields = $this->getSetting('fields');
    $custom_items = $this->getCustomFieldItems();
    foreach ($custom_items as $name => $custom_item) {
      if ($custom_item->getDataType() === 'uuid') {
        continue;
      }
      $widget = $fields[$name]['type'] ?? $custom_item->getDefaultWidget();
      $options = $this->customFieldWidgetManager->createOptionsForInstance($this->fieldDefinition->getName(), $custom_item, $widget, $fields[$name] ?? [], 'default');
      try {
        /** @var \Drupal\custom_field\Plugin\CustomFieldWidgetInterface $plugin */
        $plugin = $this->customFieldWidgetManager->getInstance($options);
        $plugins[(string) $name] = $plugin;
      }
      catch (PluginException $e) {
        // No errors applicable if we somehow have an invalid plugin.
      }
    }

    return $plugins;
  }

}
