<?php

namespace Drupal\d2;

use Composer\Composer;
use Composer\Config;
use Composer\Script\Event;
use Symfony\Component\Filesystem\Exception\IOExceptionInterface;
use Symfony\Component\Filesystem\Filesystem;

/**
 * Helper class to provide Composer scripts.
 */
class ScriptHandler {

  /**
   * The pattern of the filesystem layout within the package tarball.
   */
  const BIN_PATH_PATTERN = "d2-v%s/bin/d2";

  /**
   * The name of the D2 binary as used in the Composer 'bin' directory.
   */
  const BINARY_NAME = 'd2';

  /**
   * The name of the D2 (pseudo-)package.
   */
  const PACKAGE_NAME = 'terrastruct/d2';

  /**
   * Copy the package binary to the Composer 'bin' directory.
   */
  public static function installBinary(Event $event) {
    $composer = $event->getComposer();
    try {
      $src = self::getPackageBinaryPath($composer);
      $dest = self::getInstalledBinaryPath($composer->getConfig());
      (new Filesystem())->copy($src, $dest, TRUE);
      $event->getIO()->write("Copied D2 binary to {$dest}.");
    } catch (IOExceptionInterface $exception) {
      $event->getIO()->writeError("<error>Failed to copy D2 binary to {$dest}</error>");
    }
  }

  /**
   * Return the path to the binary in the Composer 'bin' directory.
   */
  public static function getInstalledBinaryPath(Config $config): string {
    return $config->get('bin-dir') . DIRECTORY_SEPARATOR . self::BINARY_NAME;
  }

  /**
   * Return the path to the binary in the deployed package.
   */
  protected static function getPackageBinaryPath(Composer $composer): string {
    $package = $composer
      ->getRepositoryManager()
      ->getLocalRepository()
      ->findPackage(self::PACKAGE_NAME, "*");
    $package_path = $composer
      ->getInstallationManager()
      ->getInstallPath($package);
    $bin_path = sprintf(self::BIN_PATH_PATTERN, $package->getPrettyVersion());
    return $package_path . DIRECTORY_SEPARATOR . $bin_path;
  }

}
