<?php

namespace Drupal\d2;

use Composer\Factory;
use Composer\Json\JsonFile;
use DrupalFinder\DrupalFinderComposerRuntime;
use Symfony\Component\Process\Exception\ProcessFailedException;
use Symfony\Component\Process\Process;

/**
 * Helper class to interface with the D2 CLI.
 */
class D2Helper {

  /**
   * Return an SVG for a given D2 input.
   */
  public function getSvg(string $d2_input): string {
    $cid = 'd2:' . hash('sha256', $d2_input);

    if ($cache = \Drupal::cache()->get($cid)) {
      $svg = $cache->data;
    }
    else {
      try {
        $svg = $this->generateSvg($d2_input);
        \Drupal::cache()->set($cid, $svg);
      }
      catch (ProcessFailedException $e) {
        $error = $e->getProcess()->getErrorOutput();
        \Drupal::logger('d2')->error($error);

        if (!\Drupal::currentUser()->hasPermission('view d2 diagram errors')) {
          return '';
        }
        \Drupal::messenger()->addError(t('D2 failed to generate SVG from input: <br />%error', ['%error' => $error]));
        return '<pre>' . $d2_input . '</pre>';
      }
    }

    return $svg;
  }

  /**
   * Generate an SVG from D2 input.
   */
  public function generateSvg(string $d2_input): string {
    # Use STDIN and STDOUT.
    $d2_command = [$this->d2Bin(), "-", "-"];
    $d2_command_options = \Drupal::moduleHandler()->invokeAll('d2_command_options');
    $process = new Process(array_merge($d2_command, $d2_command_options));
    $process->setInput($d2_input);
    $process->run();

    if (!$process->isSuccessful()) {
      throw new ProcessFailedException($process);
    }

    return $process->getOutput();
  }

  /**
   * Return the full path to the D2 CLI binary.
   */
  protected function d2Bin(): string {
    // @TODO: Allow use of a system-wide D2 binary.
    $composer_root = (new DrupalFinderComposerRuntime)->getComposerRoot();
    $composer_file = $composer_root . DIRECTORY_SEPARATOR . Factory::getComposerFile();
    $config = Factory::createConfig(cwd: $composer_root);
    $config->merge((new JsonFile($composer_file))->read(), $composer_file);
    return ScriptHandler::getInstalledBinaryPath($config);
  }
}
