/**
 * @file
 * Toggles the 'data-dmt-mode' attribute based on the selected "dark mode".
 */

((Drupal, once) => {
  /**
   * Attaches behavior to toggle a data attribute based on the selected mode.
   *
   * @type {Drupal~behavior}
   *
   * @prop {Drupal~behaviorAttach} attach
   *   Toggles the 'data-dmt-mode' attribute on the HTML element based on the
   *   selected "dark mode", and updates the 'data-dmt-source' attribute
   *   accordingly.
   */
  Drupal.behaviors.darkModeToggle = {
    attach(context) {
      once('darkModeToggle', 'html', context).forEach(() => {
        const container = context.querySelector('[data-dmt-container]');

        if (!container) {
          return;
        }

        container.addEventListener('click', (event) => {
          const button = event.target.closest('[data-dmt-preference]');

          if (!button) {
            return;
          }

          const preference = button.getAttribute('data-dmt-preference');

          if (preference === 'system') {
            // Whenever the user explicitly chooses to respect/switch back to
            // the OS preference.
            localStorage.removeItem('dmt-mode');
            const isModeDark = window.matchMedia(
              '(prefers-color-scheme: dark)',
            ).matches;
            document.documentElement.setAttribute(
              'data-dmt-mode',
              isModeDark ? 'dark' : 'light',
            );
            document.documentElement.setAttribute('data-dmt-source', 'system');
          } else {
            // When the user explicitly chooses the light or dark mode.
            localStorage.setItem('dmt-mode', preference);
            document.documentElement.setAttribute('data-dmt-mode', preference);
            document.documentElement.setAttribute('data-dmt-source', 'user');
          }
        });

        window
          .matchMedia('(prefers-color-scheme: dark)')
          .addEventListener('change', (e) => {
            // When the system preference changes and the user has chosen to
            // respect the system preference.
            if (
              document.documentElement.getAttribute('data-dmt-source') ===
              'system'
            ) {
              document.documentElement.setAttribute(
                'data-dmt-mode',
                e.matches ? 'dark' : 'light',
              );
            }
          });
      });
    },
  };
})(Drupal, once);
