<?php

declare(strict_types=1);

namespace Drupal\Tests\dark_mode_toggle\FunctionalJavascript;

use Drupal\FunctionalJavascriptTests\WebDriverTestBase;
use PHPUnit\Framework\Attributes\DataProvider;
use PHPUnit\Framework\Attributes\Group;

/**
 * Tests the JavaScript functionality of the Dark Mode Toggle module.
 */
#[Group('dark_mode_toggle')]
class DarkModeToggleTest extends WebDriverTestBase {

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'block',
    'user',
    'dark_mode_toggle',
  ];

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $admin_user = $this->drupalCreateUser([
      'administer site configuration',
    ]);

    // Log in as the admin user, create/place the dark mode toggle block and
    // visit the front page.
    $this->drupalLogin($admin_user);
    $this->drupalPlaceBlock('dark_mode_toggle');
    $this->drupalGet('<front>');
  }

  /**
   * Tests whether the dark mode functionality is loaded correctly.
   */
  public function testJavaScriptFilesAreLoadedCorrectly(): void {
    $assert_session = $this->assertSession();

    // To prevent FOUC, the initialization script should be in the head element.
    $assert_session->elementExists('css', 'head script[src*="dark-mode-toggle.init.js"]');

    // The additional script should be loaded in the body element.
    $assert_session->elementExists('css', 'body script[src*="dark-mode-toggle.js"]');

    // As the functionality should only be applied once, make sure once() is
    // being used.
    $assert_session->elementAttributeContains('css', 'html', 'data-once', 'darkModeToggle');
  }

  /**
   * Tests complete user journey scenarios.
   *
   * @param array $journey
   *   Array of steps in the user journey.
   * @param array $expected_final_state
   *   Expected final state after the journey.
   */
  #[DataProvider('userJourneyProvider')]
  public function testCompleteUserJourneys(array $journey, array $expected_final_state): void {
    $assert_session = $this->assertSession();

    foreach ($journey as $step) {
      if ($step['action'] === 'click') {
        $button = $assert_session->elementExists('css', '[data-dmt-preference="' . $step['preference'] . '"]');
        $button->click();

        // Ensure the expected states are applied.
        $this->assertModeAndSource($step['expected_mode'], $step['expected_source']);
      }
      elseif ($step['action'] === 'reload') {
        $this->drupalGet('<front>');
      }
    }

    // Verify final state.
    $this->assertModeAndSource($expected_final_state['mode'], $expected_final_state['source']);
  }

  /**
   * Data provider for complete user journey testing.
   *
   * @return \Generator
   *   Yields arrays of user journey scenarios.
   */
  public static function userJourneyProvider(): \Generator {
    // Basic initialization scenarios. We assume that the default
    // 'prefers-color-scheme' for the test system is set to 'light', overriding
    // the 'prefers-color-scheme' media query is not feasible in this context.
    yield 'initial load with system light preference' => [
      'journey' => [],
      'expected_final_state' => ['mode' => 'light', 'source' => 'system'],
    ];

    // User preference scenarios.
    $user_preferences = ['light', 'dark'];
    foreach ($user_preferences as $preference) {
      yield sprintf('user sets %s mode', $preference) => [
        'journey' => [
          [
            'action' => 'click',
            'preference' => $preference,
            'expected_mode' => $preference,
            'expected_source' => 'user',
          ],
        ],
        'expected_final_state' => ['mode' => $preference, 'source' => 'user'],
      ];

      yield sprintf('user sets %s mode and reloads', $preference) => [
        'journey' => [
          [
            'action' => 'click',
            'preference' => $preference,
            'expected_mode' => $preference,
            'expected_source' => 'user',
          ],
          ['action' => 'reload'],
        ],
        'expected_final_state' => ['mode' => $preference, 'source' => 'user'],
      ];

      yield sprintf('user sets %s and switches to system', $preference) => [
        'journey' => [
          [
            'action' => 'click',
            'preference' => $preference,
            'expected_mode' => $preference,
            'expected_source' => 'user',
          ],
          [
            'action' => 'click',
            'preference' => 'system',
            'expected_mode' => 'light',
            'expected_source' => 'system',
          ],
        ],
        'expected_final_state' => ['mode' => 'light', 'source' => 'system'],
      ];
    }
  }

  /**
   * Tests behavior when multiple instances interact.
   *
   * @param string $first_action
   *   The first action to take (i.e, 'light', 'dark' or 'system').
   * @param string $second_action
   *   The second action to take on a different instance (i.e., 'light', 'dark'
   *   or 'system').
   */
  #[DataProvider('multipleInstanceInteractionProvider')]
  public function testMultipleInstanceInteractions(string $first_action, string $second_action): void {
    // Place a second dark mode toggle block.
    $this->drupalPlaceBlock('dark_mode_toggle', [
      'id' => 'dark_mode_toggle_second',
      'region' => 'sidebar_first',
    ]);

    $this->drupalGet('<front>');

    $containers = $this->getSession()->getPage()->findAll('css', '[data-dmt-container]');
    $this->assertCount(2, $containers);

    // Find and click the button for the first action.
    $first_button = $containers[0]->find('css', '[data-dmt-preference="' . $first_action . '"]');
    $this->assertNotNull($first_button, sprintf("Button to select '%s' mode found for first block not found.", $first_action));
    $first_button->click();

    // Verify the expected mode and source after the first action.
    $expected_mode = ($first_action === 'system') ? 'light' : $first_action;
    $expected_source = ($first_action === 'system' ? 'system' : 'user');
    $this->assertModeAndSource($expected_mode, $expected_source);

    // Find and click the button for the second action.
    $second_button = $containers[1]->find('css', '[data-dmt-preference="' . $second_action . '"]');
    $this->assertNotNull($first_button, sprintf("Button to select '%s' mode found for second block not found.", $second_action));
    $second_button->click();

    // Verify the expected mode and source after the second action.
    $expected_mode = ($first_action === 'system') ? 'light' : $first_action;
    $expected_source = ($first_action === 'system' ? 'system' : 'user');
    $this->assertModeAndSource($expected_mode, $expected_source);
  }

  /**
   * Data provider for multiple instance interaction testing.
   *
   * @return \Generator
   *   Yields arrays of interaction scenarios.
   */
  public static function multipleInstanceInteractionProvider(): \Generator {
    yield 'dark then light' => ['dark', 'light'];
    yield 'light then dark' => ['light', 'dark'];
    yield 'dark then system' => ['dark', 'system'];
    yield 'light then system' => ['light', 'system'];
  }

  /**
   * Helper to assert current mode and source.
   *
   * @param string $expected_mode
   *   The expected mode (i.e., 'light' or 'dark').
   * @param string $expected_source
   *   The expected source (i.e., 'user' or 'system').
   */
  protected function assertModeAndSource(string $expected_mode, string $expected_source): void {
    $assert_session = $this->assertSession();

    // Wait for the HTML element to have the expected data attributes.
    $assert_session->waitForElement('css', 'html[data-dmt-mode="' . $expected_mode . '"]');
    $assert_session->waitForElement('css', 'html[data-dmt-source="' . $expected_source . '"]');

    // Assert the HTML element has the expected data attributes.
    $assert_session->elementAttributeContains('css', 'html', 'data-dmt-mode', $expected_mode);
    $assert_session->elementAttributeContains('css', 'html', 'data-dmt-source', $expected_source);
  }

}
