<?php

declare(strict_types=1);

namespace Drupal\Tests\dark_mode_toggle\Kernel;

use Drupal\KernelTests\KernelTestBase;
use PHPUnit\Framework\Attributes\DataProvider;
use PHPUnit\Framework\Attributes\Group;

/**
 * Tests the Dark Mode Toggle block and theme functionality.
 */
#[Group('dark_mode_toggle')]
class DarkModeToggleRenderTest extends KernelTestBase {

  /**
   * Modules to install.
   *
   * @var array
   */
  protected static $modules = [
    'dark_mode_toggle',
  ];

  /**
   * Tests the dark mode toggle theme implementation.
   *
   * @param array $element
   *   The render array.
   * @param string $expected
   *   The expected output.
   */
  #[DataProvider('darkModeToggleRenderingProvider')]
  public function testDarkModeToggleRendering(array $element, string $expected): void {
    $this->assertThemeOutput('dark_mode_toggle', $element, $expected);
  }

  /**
   * Data provider for testDarkModeToggleRendering.
   *
   * @return \Generator
   *   A generator yielding test cases.
   */
  public static function darkModeToggleRenderingProvider(): \Generator {
    yield 'default' => [
      [
        '#theme' => 'dark_mode_toggle',
      ],
      <<<EOD

<div data-dmt-container="">
  <ul>
    <li>
      <button data-dmt-preference="light">Light</button>
    </li>
    <li>
      <button data-dmt-preference="dark">Dark</button>
    </li>
    <li>
      <button data-dmt-preference="system">System</button>
    </li>
  </ul>
</div>

EOD,
    ];

    yield 'with id attribute' => [
      [
        '#theme' => 'dark_mode_toggle',
        '#attributes' => [
          'id' => 'custom-id',
        ],
      ],
      <<<EOD

<div id="custom-id" data-dmt-container="">
  <ul>
    <li>
      <button data-dmt-preference="light">Light</button>
    </li>
    <li>
      <button data-dmt-preference="dark">Dark</button>
    </li>
    <li>
      <button data-dmt-preference="system">System</button>
    </li>
  </ul>
</div>

EOD,
    ];

    yield 'with class attributes' => [
      [
        '#theme' => 'dark_mode_toggle',
        '#attributes' => [
          'class' => ['custom-class-one', 'custom-class-two'],
        ],
      ],
      <<<EOD

<div class="custom-class-one custom-class-two" data-dmt-container="">
  <ul>
    <li>
      <button data-dmt-preference="light">Light</button>
    </li>
    <li>
      <button data-dmt-preference="dark">Dark</button>
    </li>
    <li>
      <button data-dmt-preference="system">System</button>
    </li>
  </ul>
</div>

EOD,
    ];

    yield 'with multiple attributes' => [
      [
        '#theme' => 'dark_mode_toggle',
        '#attributes' => [
          'id' => 'custom-id',
          'class' => ['custom-class'],
        ],
      ],
      <<<EOD

<div id="custom-id" class="custom-class" data-dmt-container="">
  <ul>
    <li>
      <button data-dmt-preference="light">Light</button>
    </li>
    <li>
      <button data-dmt-preference="dark">Dark</button>
    </li>
    <li>
      <button data-dmt-preference="system">System</button>
    </li>
  </ul>
</div>

EOD,
    ];

  }

  /**
   * Ensures the dark_mode_toggle library is attached.
   */
  public function testLibraryAttachment(): void {
    $build = [
      '#theme' => 'dark_mode_toggle',
    ];

    $this->container->get('renderer')->renderRoot($build);

    $this->assertArrayHasKey('#attached', $build);
    $this->assertIsArray($build['#attached']);
    $this->assertArrayHasKey('library', $build['#attached']);
    $this->assertIsArray($build['#attached']['library']);
    $this->assertContains('dark_mode_toggle/dark-mode-toggle', $build['#attached']['library']);
  }

}
