/**
 * @file
 * Toggles the 'data-dmt-mode' attribute based on the selected "dark mode".
 */

((Drupal, once) => {
  /**
   * Initializes the listener for system/OS "dark mode" preference changes.
   *
   * @param {Document|HTMLElement} context
   *   The context in which to initialize the listener.
   */
  function initializeDarkModeToggleSystemPreference(context) {
    // Listen for changes in the system preference for dark mode. This is only
    // done once on the HTML element to avoid adding multiple listeners.
    once('dmt-system-init', 'html', context).forEach(() => {
      window
        .matchMedia('(prefers-color-scheme: dark)')
        .addEventListener('change', (e) => {
          // When the system preference changes and the user has chosen to respect
          // the system preference.
          if (
            document.documentElement.getAttribute('data-dmt-source') ===
            'system'
          ) {
            document.documentElement.setAttribute(
              'data-dmt-mode',
              e.matches ? 'dark' : 'light',
            );
          }
        });
    });
  }

  /**
   * Initializes the dark mode toggle containers.
   *
   * @param {Document|HTMLElement} context
   *   The context in which to initialize the dark mode toggle containers.
   */
  function initializeDarkModeToggleContainers(context) {
    // Find all dark mode toggles within the given context.
    once('dmt-container-init', '[data-dmt-container]', context).forEach(
      (container) => {
        container.addEventListener('click', (event) => {
          const button = event.target.closest('[data-dmt-preference]');

          if (!button) {
            return;
          }

          const preference = button.getAttribute('data-dmt-preference');

          if (preference === 'system') {
            // Whenever the user explicitly chooses to respect/switch back to
            // the system/OS preference.
            localStorage.removeItem('dmt-mode');
            const isModeDark = window.matchMedia(
              '(prefers-color-scheme: dark)',
            ).matches;
            document.documentElement.setAttribute(
              'data-dmt-mode',
              isModeDark ? 'dark' : 'light',
            );
            document.documentElement.setAttribute('data-dmt-source', 'system');
          } else {
            // When the user explicitly chooses the light or dark mode.
            localStorage.setItem('dmt-mode', preference);
            document.documentElement.setAttribute('data-dmt-mode', preference);
            document.documentElement.setAttribute('data-dmt-source', 'user');
          }
        });
      },
    );
  }

  /**
   * Attaches behavior to toggle a data attribute based on the selected mode.
   *
   * @type {Drupal~behavior}
   *
   * @prop {Drupal~behaviorAttach} attach
   *   Toggles the 'data-dmt-mode' attribute on the HTML element based on the
   *   selected "dark mode", and updates the 'data-dmt-source' attribute
   *   accordingly.
   */
  Drupal.behaviors.darkModeToggle = {
    attach(context) {
      initializeDarkModeToggleSystemPreference(context);
      initializeDarkModeToggleContainers(context);
    },
  };
})(Drupal, once);
