<?php

declare(strict_types=1);

namespace Drupal\Tests\dark_mode_toggle\FunctionalJavascript;

use Drupal\FunctionalJavascriptTests\WebDriverTestBase;
use PHPUnit\Framework\Attributes\DataProvider;
use PHPUnit\Framework\Attributes\Group;
use PHPUnit\Framework\Attributes\TestWith;

/**
 * Tests the JavaScript functionality of the Dark Mode Toggle module.
 */
#[Group('dark_mode_toggle')]
class DarkModeToggleTest extends WebDriverTestBase {

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'block',
    'user',
    'dark_mode_toggle',
  ];

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $admin_user = $this->drupalCreateUser([
      'administer site configuration',
    ]);

    // Log in as the admin user, create/place the dark mode toggle block and
    // visit the front page.
    $this->drupalLogin($admin_user);
    $this->drupalPlaceBlock('dark_mode_toggle');
    $this->drupalGet('<front>');
  }

  /**
   * Tests whether the dark mode functionality is loaded correctly.
   */
  public function testJavaScriptFilesAreLoadedCorrectly(): void {
    $assert_session = $this->assertSession();

    // To prevent FOUC, the initialization script should be in the head element.
    $assert_session->elementExists('css', 'head script[src*="dark-mode-toggle.init.js"]');

    // The additional script should be loaded in the body element.
    $assert_session->elementExists('css', 'body script[src*="dark-mode-toggle.js"]');

    // As the functionality should only be applied once, make sure once() is
    // being used correctly. An event listener for changes to the
    // 'prefers-color-scheme' media query should be registered once (using a
    // data-once attribute attached to the html element), and each block
    // containing the toggle buttons should have also an event listener once
    // (also using a data-once attribute).
    $assert_session->elementAttributeContains('css', 'html', 'data-once', 'dmt-system-init');
    $assert_session->elementAttributeContains('css', '[data-dmt-container]', 'data-once', 'dmt-container-init');

    // We assume that the default 'prefers-color-scheme' for the test system is
    // set to 'light', overriding the 'prefers-color-scheme' media query is not
    // feasible in this context.
    $this->assertModeAndSourceAttributes('system');
  }

  /**
   * Tests complete user journey scenarios.
   *
   * @param array $journey
   *   Array of steps in the user journey.
   * @param string $expected_state
   *   Expected final state after the journey (i.e., 'light', 'dark' or
   *   'system').
   */
  #[DataProvider('userJourneyProvider')]
  public function testCompleteUserJourneys(array $journey, string $expected_state): void {
    $assert_session = $this->assertSession();

    foreach ($journey as $step) {
      if ($step['action'] === 'click') {
        $button = $assert_session->elementExists('css', '[data-dmt-preference="' . $step['preference'] . '"]');
        $button->click();

        // Ensure the mode and source data attributes are correctly applied.
        $this->assertModeAndSourceAttributes($step['preference']);
      }
      elseif ($step['action'] === 'reload') {
        $this->drupalGet('<front>');
      }
    }

    // Verify the final state of the mode and source data attributes.
    $this->assertModeAndSourceAttributes($expected_state);
  }

  /**
   * Data provider for complete user journey testing.
   *
   * @return \Generator
   *   Yields arrays of user journey scenarios.
   */
  public static function userJourneyProvider(): \Generator {
    // User preference scenarios.
    $user_preferences = ['light', 'dark'];
    foreach ($user_preferences as $preference) {
      yield sprintf('user sets %s mode', $preference) => [
        'journey' => [
          [
            'action' => 'click',
            'preference' => $preference,
          ],
        ],
        'expected_state' => $preference,
      ];

      yield sprintf('user sets %s mode and reloads', $preference) => [
        'journey' => [
          [
            'action' => 'click',
            'preference' => $preference,
          ],
          ['action' => 'reload'],
        ],
        'expected_state' => $preference,
      ];

      yield sprintf('user sets %s and switches to system', $preference) => [
        'journey' => [
          [
            'action' => 'click',
            'preference' => $preference,
          ],
          [
            'action' => 'click',
            'preference' => 'system',
          ],
        ],
        'expected_state' => 'system',
      ];
    }
  }

  /**
   * Tests behavior when multiple instances interact.
   *
   * @param string $first_action
   *   The first action to take (i.e, 'light', 'dark' or 'system').
   * @param string $second_action
   *   The second action to take on a different instance (i.e., 'light', 'dark'
   *   or 'system').
   */
  #[TestWith(['dark', 'light'], 'dark then light')]
  #[TestWith(['light', 'dark'], 'light then dark')]
  #[TestWith(['dark', 'system'], 'dark then system')]
  #[TestWith(['light', 'system'], 'light then system')]
  public function testMultipleInstanceInteractions(string $first_action, string $second_action): void {
    // Place a second dark mode toggle block.
    $this->drupalPlaceBlock('dark_mode_toggle', [
      'id' => 'dark_mode_toggle_second',
      'region' => 'sidebar_first',
    ]);

    $this->drupalGet('<front>');

    // Get both block containers.
    $containers = $this->getSession()->getPage()->findAll('css', '[data-dmt-container]');
    $this->assertCount(2, $containers);

    // Find and click the button for the first action.
    $first_button = $containers[0]->find('css', '[data-dmt-preference="' . $first_action . '"]');
    $this->assertNotEmpty($first_button, sprintf("Button to select '%s' mode found for first block not found.", $first_action));
    $first_button->click();

    // Verify the mode and source data attributes after the first action.
    $this->assertModeAndSourceAttributes($first_action);

    // Find and click the button for the second action.
    $second_button = $containers[1]->find('css', '[data-dmt-preference="' . $second_action . '"]');
    $this->assertNotEmpty($second_button, sprintf("Button to select '%s' mode found for second block not found.", $second_action));
    $second_button->click();

    // Verify the mode and source data attributes after the second action.
    $this->assertModeAndSourceAttributes($second_action);
  }

  /**
   * Helper to assert current mode and source data attribute values.
   *
   * @param string $preference
   *   The expected preference (i.e., 'light', 'dark' or 'system').
   */
  protected function assertModeAndSourceAttributes(string $preference): void {
    $assert_session = $this->assertSession();

    // Determine expected mode and source based on the provided preference. If
    // the preference is 'system', we assume the system's preference is 'light'.
    // When 'light' or 'dark' is provided, the source should be user.
    $expected_mode = ($preference === 'system') ? 'light' : $preference;
    $expected_source = ($preference === 'system' ? 'system' : 'user');

    // Wait for the HTML element to have the expected data attributes.
    $assert_session->waitForElement('css', 'html[data-dmt-mode="' . $expected_mode . '"]');
    $assert_session->waitForElement('css', 'html[data-dmt-source="' . $expected_source . '"]');

    // Assert the HTML element has the expected data attributes.
    $assert_session->elementAttributeContains('css', 'html', 'data-dmt-mode', $expected_mode);
    $assert_session->elementAttributeContains('css', 'html', 'data-dmt-source', $expected_source);
  }

}
