/**
 * @file
 * Users the 'data-dmu-mode' attribute based on the selected "dark mode".
 */

((Drupal, once, drupalSettings) => {

  /**
   * Initializes the listener for system/OS "dark mode" preference changes.
   *
   * @param {Document|HTMLElement} context
   *   The context in which to initialize the listener.
   */
  function initializeDarkModeUserSystemPreference(context) {
    // Listen for changes in the system preference for dark mode. This is only
    // done once on the HTML element to avoid adding multiple listeners.
    once('dmu-system-init', 'html', context).forEach(() => {
      window
        .matchMedia('(prefers-color-scheme: dark)')
        .addEventListener('change', (e) => {
          // When the system preference changes and the user has chosen to respect
          // the system preference.
          if (
            document.documentElement.getAttribute('data-dmu-source') ===
            'system'
          ) {
            document.documentElement.setAttribute(
              'data-dmu-mode',
              e.matches ? 'dark' : 'light',
            );
          }
        });
    });
  }

  /**
   * Initializes the dark mode user settings.
   *
   * @param {Document|HTMLElement} context
   *   The context in which to initialize the dark mode user settings
   */
  function initializeDarkModeUserSettings(context) {
    once('dmu-init', 'html', context).forEach(() => {
      const settings = drupalSettings.dark_mode_user;
      if (settings === 'system') {
        // Whenever the user explicitly chooses to respect/switch back to
        // the system/OS preference.
        const isModeDark = window.matchMedia(
          '(prefers-color-scheme: dark)',
        ).matches;
        document.documentElement.setAttribute(
          'data-dmu-mode',
          isModeDark ? 'dark' : 'light',
        );
        document.documentElement.setAttribute('data-dmu-source', 'system');
      }
      else {
        // When the user explicitly chooses the light or dark mode.
        document.documentElement.setAttribute('data-dmu-mode', settings);
        document.documentElement.setAttribute('data-dmu-source', 'user');
      }
    })
  }

  /**
   * Attaches behavior to user a data attribute based on the selected mode.
   *
   * @type {Drupal~behavior}
   *
   * @prop {Drupal~behaviorAttach} attach
   *   Users the 'data-dmu-mode' attribute on the HTML element based on the
   *   selected "dark mode", and updates the 'data-dmu-source' attribute
   *   accordingly.
   */
  Drupal.behaviors.darkModeUser = {
    attach(context) {
      initializeDarkModeUserSystemPreference(context);
      initializeDarkModeUserSettings(context);
    },
  };
})(Drupal, once, drupalSettings);
