<?php

declare(strict_types=1);

namespace Drupal\dark_mode_user\Hook;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Hook\Attribute\Hook;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\user\UserDataInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Defines hooks for the Dark Mode User module.
 */
class DarkModeUserHooks {

  /**
   * @var \Drupal\Core\Session\AccountProxyInterface
   */
  protected $currentUser;

  /**
   * @var \Drupal\user\UserDataInterface
   */
  protected $userData;

  /**
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * Constructs a new YourServiceInjector object.
   *
   * @param \Drupal\Core\Session\AccountProxyInterface $currentUser
   * @param \Drupal\user\UserDataInterface $userData
   */
  public function __construct(AccountProxyInterface $currentUser, UserDataInterface $userData, ConfigFactoryInterface $configFactory) {
    $this->currentUser = $currentUser;
    $this->userData = $userData;
    $this->configFactory = $configFactory;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('current_user'),
      $container->get('user.data'),
      $container->get('config.factory'),
    );
  }

  /**
   * Implements hook_entity_extra_field_info_alter().
   */
  #[Hook('entity_extra_field_info_alter')]
  public function entityExtraFieldInfoAlter(&$info) {
    $info['user']['user']['form']['dark_mode_user'] = [
      'label' => t('Dark Mode Toogle settings'),
      'description' => t('User-related Dark Mode Toogle settings'),
      'weight' => 50,
    ];
  }

  /**
   * Implements hook_form_BASE_FORM_ID_alter().
   */
  #[Hook('form_user_form_alter')]
  public function formBaseFormIdAlter(&$form, \Drupal\Core\Form\FormStateInterface $form_state, $form_id) {
    $account = $form_state->getFormObject()->getEntity();
    if (!$this->currentUser->isAnonymous() && $this->currentUser->hasPermission('access dark mode user')) {
      $account_data = $this->userData->get('dark_mode_user', $account->id(), 'dark_mode_user');
      $form['dark_mode_user'] = [
        '#type' => 'details',
        '#title' => t('Dark mode user settings'),
        '#open' => TRUE,
        'dark_mode_user' => [
          '#type' => 'radios',
          '#default_value' => $account_data,
          '#options' => [
            'light' => t('Light'),
            'dark' => t('Dark'),
            'system' => t('System'),
          ],
        ],
      ];
      $form['actions']['submit']['#submit'][] = [$this, 'submitDarkModeUser'];
    }
  }

  /**
   * Form submit handler.
   */
  public function submitDarkModeUser($form, FormStateInterface $form_state) {
    $account = $form_state->getFormObject()->getEntity();
    $this->userData->set('dark_mode_user', $account->id(), 'dark_mode_user', $form_state->getValue('dark_mode_user'));
  }

  /**
   * Implements hook_page_attachments().
   */
  #[Hook('page_attachments')]
  public function pageAttachments(array &$attachments) {
    $settings = $this->configFactory->get('dark_mode_user.settings')->get('system_default');
    if (!$this->currentUser->isAnonymous() && $this->currentUser->hasPermission('access dark mode user')) {
      $account_data = $this->userData->get('dark_mode_user', $this->currentUser->id(), 'dark_mode_user');
      if ($account_data !== NULL) {
        $settings = $account_data;
      }
    }
    $attachments['#attached']['library'][] = 'dark_mode_user/dark-mode-user';
    $attachments['#attached']['drupalSettings']['dark_mode_user'] = $settings;
  }

}
