<?php

namespace Drupal\database_dashboard\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Database\Connection;
use Drupal\Core\Database\Database;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Returns responses for Mysql Dashboard routes.
 */
class DatabaseDashboardController extends ControllerBase {

  /**
   * The database connection.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected $connection;

  /**
   * The controller constructor.
   *
   * @param \Drupal\Core\Database\Connection $connection
   *   The database connection.
   */
  public function __construct(Connection $connection) {
    $this->connection = $connection;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static($container->get('database'));
  }

  /**
   * Builds the response.
   */
  public function dashboard() {
    $databaseName = $this->connection->getConnectionOptions()['database'];
    $con = Database::getConnection('default', 'schema');

    $data = [];
    $data['database_size'] = $this->fetchDatabaseSize($con);
    $data['tables_sizes'] = $this->fetchAndFormatTableSizes($con, $databaseName);
    $data['tables_rows'] = $this->fetchTableRows($con, $databaseName);
    $data['tables_cache_sizes'] = $this->fetchAndFormatTableSizes($con, $databaseName, TRUE);
    $data['tables_cache_rows'] = $this->fetchTableRows($con, $databaseName, TRUE);

    $build['content'] = [
      '#theme' => 'database_dashboard',
      '#data' => $data,
      '#attached' => [
        'library' => [
          'database_dashboard/dashboard',
        ],
      ],
      '#cache' => ['max-age' => 0],
    ];
    return $build;
  }

  /**
   * Query and return overall database size per schema (in GB).
   */
  private function fetchDatabaseSize($con) {
    $res = $con->query("SELECT table_schema AS \"Database name\", SUM(data_length + index_length) / 1024 / 1024 / 1024 AS \"Size (GB)\" FROM TABLES WHERE engine='InnoDB' GROUP BY table_schema;");
    return $res->fetchAllKeyed();
  }

  /**
   * Query and return table sizes, formatted MB/GB.
   */
  private function fetchAndFormatTableSizes($con, $databaseName, $onlyCache = FALSE) {
    $where = $onlyCache ? " AND TABLE_NAME like 'cache%'" : "";
    $query = "SELECT table_name AS `table`, round(((data_length + index_length) / 1024 / 1024), 2) `size` FROM information_schema.TABLES WHERE table_schema = :databaseName{$where} ORDER BY (data_length + index_length) DESC LIMIT 20;";
    $res = $con->query($query, [':databaseName' => $databaseName]);
    $raw_sizes = $res->fetchAllKeyed();
    $sizes = [];
    foreach ($raw_sizes as $table => $size_mb) {
      $sizes[$table] = self::formatMbGb($size_mb);
    }
    return $sizes;
  }

  /**
   * Query and return table rows.
   */
  private function fetchTableRows($con, $databaseName, $onlyCache = FALSE) {
    $where = $onlyCache ? " AND TABLE_NAME like 'cache%'" : "";
    $query = "SELECT TABLE_NAME, TABLE_ROWS FROM `information_schema`.`tables` WHERE `table_schema` = :databaseName{$where} order by TABLE_ROWS DESC LIMIT 20;";
    $res = $con->query($query, [':databaseName' => $databaseName]);
    return $res->fetchAllKeyed();
  }

  /**
   * Formats a size in MB as "NNN.NNN GB" or "NNN.NNN MB".
   */
  protected static function formatMbGb($size_mb): string {
    if ($size_mb >= 1000) {
      return number_format($size_mb / 1024, 2, ',', ' ') . ' GB';
    }
    else {
      return number_format($size_mb, 2, ',', ' ') . ' MB';
    }
  }

}
