<?php

declare(strict_types=1);

namespace Drupal\date_ap_style\Plugin\Field\FieldFormatter;

use Drupal\Core\Field\Attribute\FieldFormatter;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
// @todo Remove this import when datetime_range module dependency is dropped.
// phpcs:ignore Drupal.Classes.UnusedUseStatement
use Drupal\datetime_range\DateTimeRangeTrait;

/**
 * Plugin implementation of the AP style date range field formatter.
 */
#[FieldFormatter(
  id: 'daterange_ap_style',
  label: new TranslatableMarkup('AP Style'),
  field_types: [
    'daterange',
    'smartdate',
  ]
)]
class ApStyleDateRangeFieldFormatter extends ApSelectFormatterBase {

  /**
   * {@inheritdoc}
   */
  public function __construct(
    $plugin_id,
    $plugin_definition,
    $field_definition,
    array $settings,
    $label,
    $view_mode,
    array $third_party_settings,
    $date_formatter,
  ) {
    parent::__construct(
      $plugin_id,
      $plugin_definition,
      $field_definition,
      $settings,
      $label,
      $view_mode,
      $third_party_settings,
      $date_formatter
    );

    // Deprecation notice for DateTimeRangeTrait dependency.
    // The trait was used in previous versions but is no longer needed as
    // date range formatting is now handled internally.
    @trigger_error('The DateTimeRangeTrait dependency in ' . __CLASS__ . ' is deprecated in date_ap_style:2.1.0 and will be removed in date_ap_style:3.0.0. The formatter now handles date ranges internally without requiring the datetime_range module. See https://www.drupal.org/project/date_ap_style/issues/3543606', E_USER_DEPRECATED);
  }

  /**
   * {@inheritdoc}
   */
  public static function defaultSettings(): array {
    $config = \Drupal::config('date_ap_style.settings');
    return [
      'separator' => $config->get('separator') ?? 'to',
    ] + parent::defaultSettings();
  }

  /**
   * {@inheritdoc}
   */
  public function settingsForm(array $form, FormStateInterface $form_state): array {
    $elements = parent::settingsForm($form, $form_state);

    $elements['separator'] = [
      '#type' => 'select',
      '#title' => $this->t('Date range separator'),
      '#options' => [
        'endash' => $this->t('En dash'),
        'to' => $this->t('to'),
      ],
      '#default_value' => $this->getSetting('separator'),
    ];
    $elements['month_only']['#description'] = $this->t('Shows only the month (e.g., Aug.) for the date, excluding the day. Year shown as required.');

    return $elements;
  }

  /**
   * {@inheritdoc}
   */
  public function settingsSummary(): array {
    $summary = parent::settingsSummary();

    if ($this->getSetting('separator') == 'endash') {
      $summary[] = $this->t('Using en dash date range separator');
    }
    else {
      $summary[] = $this->t('Using "to" date range separator');
    }

    return $summary;
  }

  /**
   * {@inheritdoc}
   */
  public function viewElements(FieldItemListInterface $items, $langcode): array {
    return parent::buildViewElements($items, $langcode);
  }

  /**
   * {@inheritdoc}
   */
  protected function processItem($item, $options, $timezone, $langcode, $field_type): array {
    $dates = [];

    if (!empty($item->start_date) && !empty($item->end_date)) {
      $start_date = $item->start_date;
      $end_date = $item->end_date;
      $dates['start'] = $start_date->getTimestamp();
      $dates['end'] = $end_date->getTimestamp();
    }

    if ($field_type === 'smartdate') {
      if (!empty($item->value) && !empty($item->end_value)) {
        $dates['start'] = $item->value;
        $dates['end'] = $item->end_value;
      }
    }

    if (isset($dates['start']) && isset($dates['end'])) {
      return [
        '#cache' => [
          'contexts' => [
            'timezone',
          ],
        ],
        '#markup' => $this->apStyleDateFormatter->formatRange($dates, $options, $timezone, $langcode, $field_type),
      ];
    }

    return [];
  }

}
