<?php

declare(strict_types=1);

namespace Drupal\date_ap_style\Plugin\Field\FieldFormatter;

use Drupal\Core\Field\Attribute\FieldFormatter;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;

/**
 * Plugin implementation of the AP style date range field formatter.
 */
#[FieldFormatter(
  id: 'daterange_ap_style',
  label: new TranslatableMarkup('AP Style'),
  field_types: [
    'daterange',
    'smartdate',
  ]
)]
class ApStyleDateRangeFieldFormatter extends ApSelectFormatterBase {

  /**
   * {@inheritdoc}
   */
  public static function defaultSettings(): array {
    $config = \Drupal::config('date_ap_style.settings');
    return [
      'separator' => $config->get('separator') ?? 'to',
    ] + parent::defaultSettings();
  }

  /**
   * {@inheritdoc}
   */
  public function settingsForm(array $form, FormStateInterface $form_state): array {
    $elements = parent::settingsForm($form, $form_state);

    $elements['separator'] = [
      '#type' => 'select',
      '#title' => $this->t('Date range separator'),
      '#options' => [
        'to' => $this->t('to'),
        'hyphen' => $this->t('Hyphen'),
        'endash' => $this->t('En dash'),
      ],
      '#default_value' => $this->getSetting('separator'),
    ];
    $elements['month_only']['#description'] = $this->t('Shows only the month (e.g., Aug.) for the date, excluding the day. Year shown as required.');

    return $elements;
  }

  /**
   * {@inheritdoc}
   */
  public function settingsSummary(): array {
    $summary = parent::settingsSummary();

    if ($this->getSetting('separator') === 'endash') {
      $summary[] = $this->t('Using en dash date range separator');
    }
    if ($this->getSetting('separator') === 'hyphen') {
      $summary[] = $this->t('Using hyphen date range separator');
    }
    else {
      $summary[] = $this->t('Using "to" date range separator');
    }

    return $summary;
  }

  /**
   * {@inheritdoc}
   *
   * Extends parent to add separator option for date ranges.
   */
  protected function buildViewElements(FieldItemListInterface $items, $langcode): array {
    $elements = [];

    $opts = [
      'always_display_year',
      'display_day',
      'use_today',
      'cap_today',
      'display_time',
      'hide_date',
      'time_before_date',
      'use_all_day',
      'month_only',
      'display_noon_and_midnight',
      'capitalize_noon_and_midnight',
    ];

    $options = [];

    foreach ($opts as $opt) {
      if ($this->getSetting($opt)) {
        // Cast to boolean to handle form values that come through as 0/1.
        $options[$opt] = (bool) $this->getSetting($opt);
      }
    }

    // Add the separator option for date ranges (not in parent).
    $options['separator'] = $this->getSetting('separator');

    $timezone = $this->getSetting('timezone') ?: NULL;
    $field_type = $items->getFieldDefinition()->getType();

    foreach ($items as $delta => $item) {
      $elements[$delta] = $this->processItem($item, $options, $timezone, $langcode, $field_type);
    }

    return $elements;
  }

  /**
   * {@inheritdoc}
   */
  protected function processItem($item, $options, $timezone, $langcode, $field_type): array {
    $dates = [];

    if (!empty($item->start_date) && !empty($item->end_date)) {
      $start_date = $item->start_date;
      $end_date = $item->end_date;
      $dates['start'] = (int) $start_date->getTimestamp();
      $dates['end'] = (int) $end_date->getTimestamp();
    }

    if ($field_type === 'smartdate') {
      if (!empty($item->value) && !empty($item->end_value)) {
        $dates['start'] = (int) $item->value;
        $dates['end'] = (int) $item->end_value;
      }
    }

    if (isset($dates['start']) && isset($dates['end'])) {
      return [
        '#cache' => [
          'contexts' => [
            'timezone',
          ],
        ],
        '#markup' => $this->apStyleDateFormatter->formatRange($dates, $options, $timezone, $langcode, $field_type),
      ];
    }

    return [];
  }

}
