<?php

namespace Drupal\date_html_segmentation_formatter\Plugin\Field\FieldFormatter;

use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Field\FormatterBase;
use Drupal\Core\Datetime\DrupalDateTime;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Render\Markup;

/**
 * Plugin implementation of the 'date_html_segmentation_formatter' formatter.
 *
 * @FieldFormatter(
 *   id = "date_html_segmentation_formatter",
 *   label = @Translation("Date HTML Segmentation Formatter"),
 *   field_types = {
 *     "smartdate",
 *     "datetime",
 *     "daterange"
 *   }
 * )
 */
class DateHTMLSegmentationFormatter extends FormatterBase {

  /**
   * {@inheritdoc}
   */
  public static function defaultSettings() {
    return [
      'date_format' => 'd.m.Y H:i',
      'range_separator' => 'to',
      'css_prefix' => 'custom',
    ] + parent::defaultSettings();
  }

  /**
   * {@inheritdoc}
   */
  public function settingsForm(array $form, FormStateInterface $form_state) {
    $elements = [];

    $elements['css_prefix'] = [
      '#type' => 'textfield',
      '#title' => $this->t('CSS Prefix'),
      '#default_value' => $this->getSetting('css_prefix'),
      '#description' => $this->t('Enter the prefix for all css selectors, e.g. "lorem" for a css class of "lorem-day".'),
      '#size' => 10,
      '#maxlength' => 10,
    ];
    $elements['range_separator'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Date date separator'),
      '#default_value' => $this->getSetting('range_separator'),
      '#description' => $this->t('Enter the separator between the start and the end date, compare "01.01.1970 to 12.12.1970" or "01.01.1970 bis 12.12.1970"'),
      '#size' => 10,
      '#maxlength' => 10,
    ];
    $elements['date_format'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Custom date format'),
      '#default_value' => $this->getSetting('date_format'),
      '#description' => $this->t('Enter a PHP date format string. Examples: "d.m.Y" for 31.12.2025, "d.M.Y" for 31.Dec.2025, "H:i" for 14:30, or "d.m.Y H:i" for both.'),
      '#size' => 10,
      '#maxlength' => 50,
    ];

    return $elements;
  }

  /**
   * {@inheritdoc}
   */
  public function viewElements(FieldItemListInterface $items, $langcode) {
    $elements = [];
    $date_format = $this->getSetting('date_format') ?: 'd.m.Y H:i';
    if (!$date_format) {
      $date_format = 'd.m.Y H:i';
    }

    $css_prefix = $this->getSetting('css_prefix') ?: 'custom';
    $format_map = [
      'd' => $css_prefix . '-day',
      'j' => $css_prefix . '-day',
      'm' => $css_prefix . '-month',
      'M' => $css_prefix . '-month',
      'F' => $css_prefix . '-month',
      'Y' => $css_prefix . '-year',
      'y' => $css_prefix . '-year',
      'H' => $css_prefix . '-hour',
      'G' => $css_prefix . '-hour',
      'i' => $css_prefix . '-minute',
      's' => $css_prefix . '-second',
    ];

    foreach ($items as $delta => $item) {
      if ($item->value) {
        $start_dt = DrupalDateTime::createFromTimestamp($item->value);
        $end_dt = $item->end_value ? DrupalDateTime::createFromTimestamp($item->end_value) : NULL;

        $start_css_additional = $this->isPastDate($start_dt) ? 'is-past' : '';

        $output = '<div class="' . $css_prefix . '-date-wrapper">';
        $output .= '<span class="' . $css_prefix . '-start-date ' . $start_css_additional . '">';
        $output .= $this->renderDateWithSpans($start_dt, $date_format, $format_map);
        $output .= '</span>';

        if ($end_dt && $end_dt->getTimestamp() !== $start_dt->getTimestamp()) {

          $range_separator = $this->getSetting('range_separator') ?: 'to';
          $output .= '<span class="' . $css_prefix . '-range-separator">&nbsp;' . $range_separator . '&nbsp;</span>';

          $end_css_additional = $this->isPastDate($end_dt) ?? 'is-past';
          $output .= '<span class="' . $css_prefix . '-end-date ' . $end_css_additional . '">';
          $output .= $this->renderDateWithSpans($end_dt, $date_format, $format_map);
          $output .= '</span>';
        }

        $output .= '</div>';
      }
      else {
        $output = '<span class="' . $css_prefix . '-date-wrapper">No date given</span>';
      }

      $elements[$delta] = ['#markup' => Markup::create($output)];
    }

    return $elements;
  }

  /**
   * @param \Drupal\Core\Datetime\DrupalDateTime $timestamp
   *   The timestamp to check.
   * @return bool
   *    If the date is in the past or not
   */
  public function isPastDate(DrupalDateTime $timestamp) {
    $now = \Drupal::time()->getRequestTime();
    return $timestamp->getTimestamp() < $now;
  }

  /**
   * Render a DrupalDateTime into individual <span> elements per segment.
   *
   * @param \Drupal\Core\Datetime\DrupalDateTime $dt
   *   The datetime.
   * @param string $format
   *   The desired format.
   * @param array $format_map
   *   The mapping from date-time segment to css selector.
   *
   * @return string
   *   HTML string with <span> elements for each segment.
   */
  public function renderDateWithSpans(DrupalDateTime $dt, string $format, array $format_map): string {
    $output = '';
    $length = strlen($format);

    for ($i = 0; $i < $length; $i++) {
      $char = $format[$i];

      if ($char === '\\') {
        $i++;
        if ($i < $length) {
          $output .= htmlspecialchars($format[$i]);
        }
        continue;
      }

      if (isset($format_map[$char])) {
        $class = $format_map[$char];
        $output .= '<span class="' . $class . '">' . $dt->format($char) . '</span>';
      }
      else {
        $output .= htmlspecialchars($char);
      }
    }

    return $output;
  }

}
