<?php

namespace Drupal\date_pager;

use Drupal\Core\Link;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\user\Entity\User;

/**
 * Extends the DateTime PHP-Class.
 *
 * Stores and applies pager granularity in a date format and
 * implements the link generation with the given granularity.
 */
class PagerDate extends \DateTime {

  use StringTranslationTrait;

  /**
   * String representation of a granularity.
   *
   * @var bool|string
   */
  public $granularity;

  /**
   * Numerical index of a granularity.
   *
   * @var false|int
   */
  public $granularityId;

  /**
   * Date pattern for date interval.
   *
   * @var string[]
   */
  protected static $datePatterns = [
    0 => 'Y-01-01\\T00:00:00',
    1 => 'Y-m-01\\T00:00:00',
    2 => 'Y-m-d\\T00:00:00',
    3 => 'Y-m-d\\TH:00:00',
    4 => 'Y-m-d\\TH:i:00',
    5 => 'Y-m-d\\TH:i:s',
  ];

  /**
   * Date interval word.
   *
   * @var string[]
   */
  protected static $granularityWord = [
    0 => 'year',
    1 => 'month',
    2 => 'day',
    3 => 'hour',
    4 => 'minute',
    5 => 'second',
  ];

  /**
   * Date pattern width.
   *
   * @var string[]
   */
  protected static $dateWidth = ['YYYY', '-MM', '-DD', 'T00', ':00', ':00'];

  /**
   * Constructs the PagerDate.
   *
   * @param string $time
   *   Any date string.
   * @param null|string $granularity
   *   Date format, eg. 'Y-m-d'.
   * @param bool $withUserTimezone
   *   Create with user timezone or UTC.
   */
  public function __construct($time, $granularity = NULL, $withUserTimezone = FALSE) {
    $this->granularity = $granularity;
    // Sometimes the given parameter doesn't match the granularity wanted.
    // We need to adjust the granularity to the given value.
    // Doesn't work with timestring like 'NOW' etc, but doesn't have to.
    if (is_null($granularity)) {
      $this->granularity = $this->findGranularity($time);
    }
    $this->granularityId = $this->findGranularityId();

    $fillUpTime = self::fillUpTimestring(substr($time, 0, $this->findGranularityLength($this->granularityId)));
    if ($withUserTimezone) {
      $fillUpTime = str_replace('UTC', self::userTimezone(), $fillUpTime);
    }
    parent::__construct($fillUpTime);
  }

  /**
   * Find the longest working format string.
   *
   * Checks if any part of Y-m-dTH:i:s works.
   * So it doesn't work for TEXT date strings like NOW or year+1
   * (but also it's not supposed to).
   *
   * @param string $dateString
   *   Any date string.
   *
   * @return string|bool
   *   The longest working part of the DateTime format string.
   */
  private function findGranularity($dateString) {
    $date_parts = ['Y', '-m', '-d', '\TH', ':i', ':s'];
    // Check if what form at works, going longest to shortest.
    for ($i = 0; $i <= 6; $i++) {
      $format = implode('', array_slice(self::$dateWidth, 0, $i + 1));
      if (strlen($format) == strlen($dateString)) {
        return implode('', array_slice($date_parts, 0, $i + 1));
      }
    }
    return FALSE;
  }

  /**
   * Helper function to take a date string and determine its granularity index.
   *
   * @return false|int
   *   Granularity index if successful, false if not.
   */
  private function findGranularityId() {
    $date_parts = ['Y', '-m', '-d', '\TH', ':i', ':s'];
    // Check what format works, going longest to shortest.
    for ($i = 5; $i >= 0; $i--) {
      if ($this->granularity == implode('', array_slice($date_parts, 0, $i + 1))) {
        return $i;
      }
    }
    return FALSE;
  }

  /**
   * Helper function to find granularity length.
   *
   * @return int
   *   Granularity length.
   */
  private function findGranularityLength($granularity) {
    return strlen(implode(array_slice(self::$dateWidth, 0, $granularity + 1)));
  }

  /**
   * Return date formatted with the granularity.
   *
   * @return string
   *   Formatted date
   */
  public function __toString() {
    return $this->format($this->granularity);
  }

  /**
   * Return unix seconds with granularity applied.
   *
   * @parameter string $format
   *   If given, it applies a format other then the own granularity.
   *
   * @return int
   *   unix seconds
   */
  public function toTime() {
    return $this->format('U');
  }

  /**
   * Checks if a date is between two others.
   *
   * It's important to apply the given granularity here.
   *
   * @parameter PagerDate $startdate
   * @parameter PagerDate $enddate
   *
   * @return bool
   *   Returns TRUE if in between.
   */
  public function between(PagerDate $startdate, PagerDate $enddate) {
    return (($this->toTime() >= $startdate->toTime()) && ($this->toTime() <= $enddate->toTime()));
  }

  /**
   * Start date depending on granularity.
   *
   * @param null|int $granularityId
   *   Granularity.
   *
   * @return string
   *   Start date string.
   */
  public function startDate($granularityId = NULL) {
    $granularityId = $granularityId ?? $this->granularityId;
    $startdate = clone $this;
    return $startdate->format(self::$datePatterns[5]);
  }

  /**
   * End date depending on granularity.
   *
   * @param null|int $granularityId
   *   Granularity.
   *
   * @return string
   *   End date string.
   */
  public function endDate($granularityId = NULL) {
    $granularityId = $granularityId ?? $this->granularityId;
    $enddate = clone $this;
    return $enddate->modify('+1 ' . self::$granularityWord[$granularityId])->format(self::$datePatterns[5]);
  }

  /**
   * Fills up time string.
   *
   * @param string $timestring
   *   Time string.
   * @param string $full_timestring
   *   Full time string.
   *
   * @return string
   *   new time string
   */
  public static function fillUpTimestring($timestring, $full_timestring = '2000-01-01T00:00:00 UTC') {
    // We have to expand the timestring, otherwise 2000 as a year
    // wouldn't be properly recognized.
    // Also we don't want to use the current date
    // as defaults for the missing parts.
    if (strlen($full_timestring) > strlen($timestring)) {
      $timestring .= substr($full_timestring, -(strlen($full_timestring) - strlen($timestring)));
    }
    return $timestring;
  }

  /**
   * Gets the time zone based on site and user configuration.
   *
   * @return string|null
   *   The time zone, or NULL if nothing is set.
   */
  public static function userTimezone() {
    $config = \Drupal::config('system.date');
    $current_user = User::load(\Drupal::currentUser()->id());
    if ($config->get('timezone.user.configurable') && $current_user->isAuthenticated() && $current_user->getTimezone()) {
      return $current_user->getTimeZone();
    }
    elseif ($default_timezone = $config->get('timezone.default')) {
      return $default_timezone;
    }
    return NULL;
  }

  /**
   * Generates pager link object.
   *
   * @param string $route_name
   *   Route to link to.
   * @param PagerDate $active_date
   *   Currently active Pager Date for comparison.
   *
   * @return array
   *   Link render array
   */
  public function toLink($route_name, PagerDate $active_date) {

    $classes = [];

    $time = $this->toTime();

    // Link text and class only the last part of the format.
    $linkFormat = substr($this->granularity, -1, 1);

    $text = $this->t(
      '<time datetime="@datetime">@linktext</time>', [
        '@datetime' => date($this->granularity, $time),
        '@linktext' => date($linkFormat, $time),
      ]
    );

    $classes[] = $linkFormat;

    // Link query parameter.
    $args['date'] = date($this->granularity, $time);

    if ($this->toTime() > time()) {
      $classes[] = 'future';
    }

    if ($this->toTime() == $active_date->toTime()) {
      $classes[] = 'active';
    }
    $options['query'] = $args;
    // Generate & return Link Object.
    return [
      'title' => $text,
      'attributes' => [
        'class' => implode(' ', $classes),
        'title' => date($linkFormat, $time),
      ],
      'url' => Link::createFromRoute($text, $route_name, [], $options)
        ->getUrl(),
      '#theme' => 'link',
    ];
  }

}
