<?php

declare(strict_types=1);

namespace Drupal\date_point\Data;

use DateTimeImmutable as DTI;

/**
 * Represents a date.
 */
final readonly class Date implements \Stringable, \JsonSerializable {

  private const string STORAGE_FORMAT = 'Y-m-d';
  private const string DISPLAY_FORMAT = self::STORAGE_FORMAT;

  /**
   * {@selfdoc}
   */
  private string $value;

  /**
   * @throws \DateMalformedStringException
   */
  public function __construct(string $value) {
    $dti = DTI::createFromFormat(self::STORAGE_FORMAT, $value);
    if (!$dti) {
      throw new \DateMalformedStringException(\sprintf('Wrong date value "%s".', $value));
    }

    $this->value = $dti->format(self::STORAGE_FORMAT);
    // Use strict validation. Dates like '2024-9-1' are not valid.
    if ($value !== $this->value) {
      throw new \DateMalformedStringException(\sprintf('Wrong date value "%s".', $value));
    }
  }

  /**
   * @throws \DateMalformedStringException
   */
  public static function fromDateTime(DTI $datetime): self {
    return new self($datetime->format(self::STORAGE_FORMAT));
  }

  /**
   * {@selfdoc}
   */
  public function format(string $format): string {
    return $this->toDateTime()->format($format);
  }

  /**
   * {@inheritdoc}
   */
  public function __toString(): string {
    return $this->format(self::DISPLAY_FORMAT);
  }

  /**
   * {@inheritdoc}
   */
  public function jsonSerialize(): string {
    return $this->format(self::DISPLAY_FORMAT);

  }

  /**
   * {@selfdoc}
   */
  public function toDateTime(?\DateTimeZone $timezone = NULL): DTI {
    return new DTI($this->value, $timezone);
  }

  /**
   * @deprecated
   */
  public function toStorage(): string {
    return (string) $this;
  }

}
