<?php

declare(strict_types=1);

namespace Drupal\date_point\Data;

use DateTimeImmutable as DTI;

/**
 * A date-time without a time-zone in the ISO-8601 calendar system.
 *
 * @cspell:ignore VVVUUU
 */
final readonly class LocalDateTime implements \Stringable, \JsonSerializable {

  private const string STORAGE_FORMAT = 'Y-m-d H:i:s.u';
  private const string DISPLAY_FORMAT = 'Y-m-d H:i:s';

  /**
   * @var non-empty-string
   */
  private string $value;

  /**
   * {@selfdoc}
   *
   * @throws \DateMalformedStringException
   */
  public function __construct(string $input) {
    // -- Normalize input value.
    $input_length = \strlen($input);
    // "YYYY-MM-DD HH:II:SS" (19 chars) → Valid, adds microseconds.
    if ($input_length === 19) {
      $value = $input . '.000000';
    }
    // "YYYY-MM-DD HH:II:SS.VVVUUU" (21-26 chars) → Valid, pads with zeros.
    elseif ($input_length >= 21 && $input_length <= 26) {
      $value = \str_pad($input, 26, '0');
    }
    // Any other lengths are invalid.
    else {
      self::throwException($input);
    }

    // -- Validate format strictly.
    if (!$dti = DTI::createFromFormat(self::STORAGE_FORMAT, $value)) {
      self::throwException($input);
    }
    // DateTimeImmutable::createFromFormat() has loose validation. For example,
    // it accepts "1-1-4 1:00:00" as valid for our format. Therefore, we compare
    // the normalized value with the formatted result for exact match.
    $this->value = $dti->format(self::STORAGE_FORMAT);
    if ($value !== $this->value) {
      self::throwException($input);
    }
  }

  /**
   * Converts DateTimeImmutable to LocalDateTime by omitting timezone.
   *
   * The input datetime must already be in the desired local timezone.
   */
  public static function fromDateTime(DTI $datetime): self {
    return new self($datetime->format(self::STORAGE_FORMAT));
  }

  /**
   * Creates local datetime from Unix timestamp.
   *
   * @throws \DateMalformedStringException
   */
  public static function fromTimestamp(int|float $timestamp): self {
    try {
      return self::fromDateTime(
        // @todo Use DateTimeImmutable::createFromTimestamp() once we drop
        // support for PHP 8.3.
        new \DateTimeImmutable('@' . $timestamp),
      );
    }
    catch (\DateMalformedStringException $exception) {
      throw new \DateMalformedStringException(
        \sprintf('Wrong timestamp "%s".', $timestamp), previous: $exception,
      );
    }
  }

  /**
   * Converts the datetime to Unix timestamp.
   */
  public function toTimestamp(): float {
    return (float) $this->format('U.u');
  }

  /**
   * {@selfdoc}
   */
  public function format(string $format): string {
    return $this->toDateTime(new \DateTimeZone('UTC'))->format($format);
  }

  /**
   * {@inheritdoc}
   */
  public function __toString(): string {
    return $this->format(self::DISPLAY_FORMAT);
  }

  /**
   * {@inheritdoc}
   */
  public function jsonSerialize(): string {
    return $this->format(self::DISPLAY_FORMAT);
  }

  /**
   * {@selfdoc}
   */
  public function toDateTime(?\DateTimeZone $timezone = NULL): DTI {
    return new DTI($this->value, $timezone);
  }

  /**
   * @throws \DateMalformedStringException
   */
  private static function throwException(string $datetime): never {
    throw new \DateMalformedStringException(\sprintf('Wrong datetime value "%s".', $datetime));
  }

}
