<?php

declare(strict_types=1);

namespace Drupal\date_point\Data;

use DateTimeImmutable as DTI;

/**
 * Represents a time point without time zone.
 *
 * Note that this object should only be used for time points. It is not suitable
 * to represent durations, i.e. travel time and so on.
 *
 * @cspell:ignore VVVUUU
 */
final readonly class Time implements \Stringable, \JsonSerializable {

  private const string STORAGE_FORMAT = 'H:i:s.u';
  private const string DISPLAY_FORMAT = 'H:i:s';

  /**
   * {@selfdoc}
   */
  private string $value;

  /**
   * {@selfdoc}
   */
  public function __construct(string $input) {
    // -- Normalize input value.
    $input_length = \strlen($input);

    // - "HH:II" (5 chars) → Valid, adds seconds and microseconds.
    if ($input_length === 5) {
      $value = $input . ':00.000000';
    }
    // - "HH:II:SS" (8 chars) → Valid, adds microseconds.
    elseif ($input_length === 8) {
      $value = $input . '.000000';
    }
    // - "HH:II:SS.VVVUUU" (10-15 chars) → Valid, pads with zeros.
    elseif ($input_length >= 10 && $input_length <= 15) {
      $value = \str_pad($input, 15, '0');
    }
    // - Any other lengths invalid.
    else {
      self::throwException($input);
    }

    $datetime = DTI::createFromFormat(self::STORAGE_FORMAT, $value);
    if (!$datetime) {
      self::throwException($input);
    }
    $this->value = $datetime->format(self::STORAGE_FORMAT);

    // Unlike PHP DateTime object this one does not allow overflows, which means
    // it is not possible to set time to something like 25:61:61.
    if ($value !== $this->value) {
      self::throwException($input);
    }
  }

  /**
   * Converts DateTimeImmutable to Time by omitting timezone.
   *
   * The input datetime must already be in the desired local timezone.
   */
  public static function fromDateTime(DTI $datetime): self {
    return new self($datetime->format(self::STORAGE_FORMAT));
  }

  /**
   * {@selfdoc}
   */
  public static function fromSeconds(int|float $seconds): self {
    if ($seconds < 0 || $seconds >= 86400) {
      throw new \OutOfRangeException('The time value should be in withing 0 - 86400 seconds.');
    }
    $seconds_part = (int) \floor($seconds);
    $microseconds_part = (int) (1_000_000 * ($seconds - $seconds_part));
    // @todo Use DateTimeImmutable::createFromTimestamp() once we drop support
    // for PHP 8.3.
    $datetime = (new DTI())->setTime(0, 0, $seconds_part, $microseconds_part);
    return self::fromDateTime($datetime);
  }

  /**
   * {@selfdoc}
   */
  public function format(string $format): string {
    return $this->getDateTime()->format($format);
  }

  /**
   * {@inheritdoc}
   */
  public function __toString(): string {
    return $this->format(self::DISPLAY_FORMAT);
  }

  /**
   * {@inheritdoc}
   */
  public function jsonSerialize(): string {
    return $this->format(self::DISPLAY_FORMAT);
  }

  /**
   * @deprecated
   */
  public function toStorage(): string {
    return $this->value;
  }

  /**
   * @throws \DateMalformedStringException
   */
  private static function throwException(string $time): never {
    throw new \DateMalformedStringException(\sprintf('Wrong time value "%s".', $time));
  }

  /**
   * {@selfdoc}
   */
  private function getDateTime(): DTI {
    return new DTI($this->value);
  }

}
