<?php

declare(strict_types=1);

namespace Drupal\date_point\Data;

use DateTimeImmutable as DTI;

/**
 * Represents a month of a specific year.
 */
final readonly class YearMonth implements \Stringable, \JsonSerializable {

  private const string STORAGE_FORMAT = 'Y-m';
  private const string DISPLAY_FORMAT = 'Y-m';

  /**
   * {@selfdoc}
   */
  private string $value;

  /**
   * {@selfdoc}
   */
  public function __construct(string $value) {
    // @see https://stackoverflow.com/questions/36307921/php-createfromfromformat-month-year-only
    $dti = DTI::createFromFormat(self::STORAGE_FORMAT . '-d', $value . '-01');
    if (!$dti) {
      throw new \DateMalformedStringException(\sprintf('Wrong month value "%s".', $value));
    }

    $this->value = $dti->format(self::STORAGE_FORMAT);
    // Use strict validation. Dates month '2024-9' are not valid.
    if ($value !== $this->value) {
      throw new \DateMalformedStringException(\sprintf('Wrong month value "%s".', $value));
    }
  }

  /**
   * {@selfdoc}
   */
  public static function fromDateTime(DTI $datetime): self {
    return new self($datetime->format(self::STORAGE_FORMAT));
  }

  /**
   * {@selfdoc}
   */
  public function toDateTime(?\DateTimeZone $timezone = NULL): DTI {
    return new DTI($this->value, $timezone);
  }

  /**
   * {@selfdoc}
   */
  public function format(string $format): string {
    return $this->toDateTime()->format($format);
  }

  /**
   * {@inheritdoc}
   */
  public function __toString(): string {
    return $this->format(self::DISPLAY_FORMAT);
  }

  /**
   * {@inheritdoc}
   */
  public function jsonSerialize(): string {
    return $this->format(self::DISPLAY_FORMAT);
  }

  /**
   * Defines timestamp boundaries suitable for SQL queries.
   *
   * These boundaries only suitable for querying Date Point timestamps.
   *
   * @phpstan-return array{0: non-empty-string, 1: non-empty-string}
   *
   * @see \Drupal\date_point\Data\DateTime\Granularity::getBoundaries
   */
  public function getBoundaries(): array {
    $start = $this->toDateTime();
    $end = $start->modify('+1 month');
    $format = 'Y-m-d 00:00:00.000000';
    return [$start->format($format), $end->format($format)];
  }

}
