<?php

declare(strict_types=1);

namespace Drupal\Tests\date_point\Unit;

use DateTimeImmutable as DTI;
use Drupal\date_point\Data\LocalDateTime;
use Drupal\Tests\UnitTestCase;

/**
 * A test for local datetime value object.
 *
 * @group date_point
 */
final class LocalDateTimeTest extends UnitTestCase {

  /**
   * @dataProvider localDateTimeProvider
   */
  public function testConstructor(string $input, ?string $expected_datetime): void {
    if (!$expected_datetime) {
      self::expectExceptionObject(
        new \DateMalformedStringException(\sprintf('Wrong datetime value "%s".', $input)),
      );
    }
    $local_datetime = new LocalDateTime($input);
    self::assertSame($expected_datetime, $local_datetime->format('Y-m-d H:i:s.u'));
  }

  /**
   * {@selfdoc}
   */
  public static function localDateTimeProvider(): \Generator {
    yield ['123', NULL];
    yield ['wrong', NULL];
    yield ['2024 - 12-05', NULL];
    yield [' 2024-12-05 ', NULL];
    yield ['2024-12-05', NULL];
    yield ['0000-12-05', NULL];
    yield ['0000-1-5', NULL];
    yield ['12345-1-5', NULL];
    yield ['2024-12-05T00:12:15', NULL];
    yield ['2024-12-05 00', NULL];
    yield ['2024-12-05 00:12', NULL];
    yield ['2024-12-05 00:12:15', '2024-12-05 00:12:15.000000'];
    yield [' 2024-12-05 00:12:15', NULL];
    yield ['2024-12-05 00:12:15 ', NULL];
    yield ['2024-13-05 00:12:15', NULL];
    yield ['2024-12-32 00:12:15', NULL];
    yield ['2024-12-05 24:12:15', NULL];
    yield ['2024-12-05 15:31:45.', NULL];
    yield ['2024-12-05 15:32:45', '2024-12-05 15:32:45.000000'];
    yield ['2024-12-05 15:33:45.1', '2024-12-05 15:33:45.100000'];
    yield ['2024-12-05 15:34:45.12', '2024-12-05 15:34:45.120000'];
    yield ['2024-12-05 15:35:45.123', '2024-12-05 15:35:45.123000'];
    yield ['2024-12-05 15:36:45.1234', '2024-12-05 15:36:45.123400'];
    yield ['2024-12-05 15:37:45.12345', '2024-12-05 15:37:45.123450'];
    yield ['2024-12-05 15:38:45.123456', '2024-12-05 15:38:45.123456'];
    yield ['2024-12-05 15:39:45.1234567', NULL];
    yield ['2024-12-05 15:40:45.abc', NULL];
  }

  /**
   * {@selfdoc}
   */
  public function testFromDateTime(): void {
    $local_datetime = LocalDateTime::fromDateTime(
      new DTI('2024-01-01 15:49:51.123', new \DateTimeZone('Europe/Moscow')),
    );
    self::assertSame('2024-01-01 15:49:51.123000', $local_datetime->format('Y-m-d H:i:s.u'));

    $local_datetime = LocalDateTime::fromDateTime(
      new DTI('2024-01-05', new \DateTimeZone('UTC')),
    );
    self::assertSame('2024-01-05 00:00:00.000000', $local_datetime->format('Y-m-d H:i:s.u'));
  }

  /**
   * {@selfdoc}
   */
  public function testFormat(): void {
    $local_datetime = new LocalDateTime('2000-04-01 12:15:01.123456');
    self::assertSame('01.04.2000', $local_datetime->format('d.m.Y'));
  }

  /**
   * {@selfdoc}
   */
  public function testToString(): void {
    $local_datetime = new LocalDateTime('2000-04-01 12:15:01.123456');
    self::assertSame('2000-04-01 12:15:01', (string) $local_datetime);
  }

  /**
   * {@selfdoc}
   */
  public function testJsonSerialize(): void {
    $local_datetime = new LocalDateTime('2000-04-01 12:15:01.123');
    self::assertSame('"2000-04-01 12:15:01"', \json_encode($local_datetime));
  }

  /**
   * {@selfdoc}
   */
  public function testToDatetime(): void {
    $local_datetime = new LocalDateTime('2000-04-01 12:15:01.123');
    self::assertEquals(
      new \DateTimeImmutable('2000-04-01 12:15:01.123'),
      $local_datetime->toDateTime(),
    );
    self::assertEquals(
      new \DateTimeImmutable('2000-04-01 12:15:01.123', new \DateTimeZone('Europe/Moscow')),
      $local_datetime->toDateTime(new \DateTimeZone('Europe/Moscow')),
    );
    self::assertEquals(
      new \DateTimeImmutable('2000-04-01 12:15:01.123', new \DateTimeZone('Asia/Shanghai')),
      $local_datetime->toDateTime(new \DateTimeZone('Asia/Shanghai')),
    );
  }

  /**
   * @dataProvider fromTimestampProvider
   */
  public function testFromTimestamp(int|float $timestamp, LocalDateTime $expected_datetime): void {
    $actual_datetime = LocalDateTime::fromTimestamp($timestamp);
    self::assertEquals($expected_datetime, $actual_datetime);
  }

  /**
   * {@selfdoc}
   */
  public static function fromTimestampProvider(): \Generator {
    yield [-600, new LocalDateTime('1969-12-31 23:50:00.000000')];
    yield [0, new LocalDateTime('1970-01-01 00:00:00.000000')];
    yield [900, new LocalDateTime('1970-01-01 00:15:00.000000')];
    yield [900.654321, new LocalDateTime('1970-01-01 00:15:00.654321')];
  }

  /**
   * @dataProvider toTimestampProvider
   */
  public function testToTimestamp(LocalDateTime $datetime, float $expected_timestamp): void {
    $actual_timestamp = $datetime->toTimestamp();
    self::assertEquals($expected_timestamp, $actual_timestamp);
  }

  /**
   * {@selfdoc}
   */
  public static function toTimestampProvider(): \Generator {
    yield [new LocalDateTime('1969-12-31 23:00:00.000000'), -3600.0];
    yield [new LocalDateTime('1970-01-01 00:00:00.000000'), 0.0];
    yield [new LocalDateTime('1970-01-01 01:00:00.000000'), 3600.0];
    yield [new LocalDateTime('1970-01-01 01:00:00.123456'), 3600.123456];
  }

}
