<?php

declare(strict_types=1);

namespace Drupal\drpw_field\Plugin\DateRangePicker\ViewsOperator;

use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\drpw_field\Attribute\DateRangePickerViewsOperator;
use Drupal\drpw_field\DateRangePickerViewsOperatorPluginBase;
use Drupal\views\Plugin\views\query\Sql as ViewsQuerySql;
use Drupal\Component\Serialization\Json;

/**
 * Provides a plugin for the "In Future" operator.
 */
#[DateRangePickerViewsOperator(
  id: 'daterangepicker_views_operator_in_future',
  label: new TranslatableMarkup('In Future'),
  description: new TranslatableMarkup('Field date range is in the future of the user selected date range.'),
  short_name: new TranslatableMarkup('in_future'),
  values: 1,
)]
final class InFuture extends DateRangePickerViewsOperatorPluginBase {

  /**
   * Logic for the in_future operator.
   *
   * @param \Drupal\views\Plugin\views\query\Sql $query
   *   Views query plugin for an SQL query.
   * @param int $group
   *   The filter group.
   * @param int $position
   *   The operator position.
   * @param array $options
   *   Views filter options.
   * @param string $field
   *   The fully qualified field name.
   *   $field value is in the future of the range value in the Value field.
   * @param mixed $value
   *   The value selected in the backend interface for this operator or the
   *   user selected value in case of an exposed filter.
   */
  public function method(ViewsQuerySql $query, int $group, int $position, array $options, string $field, mixed $value): void {
    $value = is_array($value) ? reset($value) : $value;
    if (!empty($value)) {
      ['end' => $end] = Json::decode($value);
      $end_placeholder = ":end_{$position}";

      $query->addWhereExpression($group, "JSON_UNQUOTE(JSON_EXTRACT({$field}, '$.start')) > {$end_placeholder}", [$end_placeholder => $end]);
    }
  }

}
