<?php

declare(strict_types=1);

namespace Drupal\drpw_field\Plugin\DateRangePicker\ViewsOperator;

use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\drpw_field\Attribute\DateRangePickerViewsOperator;
use Drupal\drpw_field\DateRangePickerViewsOperatorPluginBase;
use Drupal\views\Plugin\views\query\Sql as ViewsQuerySql;
use Drupal\Component\Serialization\Json;

/**
 * Provides a plugin for the "Overlaps" operator.
 */
#[DateRangePickerViewsOperator(
  id: 'daterangepicker_views_operator_overlaps',
  label: new TranslatableMarkup('Overlaps'),
  description: new TranslatableMarkup('Field date range overlaps the user selected date range.'),
  short_name: new TranslatableMarkup('overlaps'),
  values: 1,
)]
final class Overlaps extends DateRangePickerViewsOperatorPluginBase {

  /**
   * Logic for the overlaps operator.
   *
   * @param \Drupal\views\Plugin\views\query\Sql $query
   *   Views query plugin for an SQL query.
   * @param int $group
   *   The filter group.
   * @param int $position
   *   The operator position.
   * @param array $options
   *   Views filter options.
   * @param string $field
   *   The fully qualified field name.
   *   $field value overlaps the range value in the Value field.
   * @param mixed $value
   *   The value selected in the backend interface for this operator or the
   *   user selected value in case of an exposed filter.
   */
  public function method(ViewsQuerySql $query, int $group, int $position, array $options, string $field, mixed $value): void {
    $value = is_array($value) ? reset($value) : $value;
    if (!empty($value)) {
      ['start' => $start, 'end' => $end] = Json::decode($value);
      $connection = $query->getConnection();
      $start_placeholder = ":start_{$position}";
      $end_placeholder = ":end_{$position}";

      // Field overlaps Value on right.
      $and_cond1 = ($connection->condition('AND'))
        ->where("JSON_UNQUOTE(JSON_EXTRACT({$field}, '$.start')) >= {$start_placeholder}", [$start_placeholder => $start])
        ->where("JSON_UNQUOTE(JSON_EXTRACT({$field}, '$.start')) <= {$end_placeholder}", [$end_placeholder => $end]);

      // Field overlaps Value on left.
      $and_cond2 = ($connection->condition('AND'))
        ->where("JSON_UNQUOTE(JSON_EXTRACT({$field}, '$.end')) >= {$start_placeholder}", [$start_placeholder => $start])
        ->where("JSON_UNQUOTE(JSON_EXTRACT({$field}, '$.end')) <= {$end_placeholder}", [$end_placeholder => $end]);

      $or_cond1 = ($connection->condition('OR'))
        ->condition($and_cond1)
        ->condition($and_cond2);

      // Value overlaps field on the left.
      $and_cond3 = ($connection->condition('AND'))
        ->where("{$end_placeholder} >= JSON_UNQUOTE(JSON_EXTRACT({$field}, '$.start'))", [$end_placeholder => $end])
        ->where("{$end_placeholder} <= JSON_UNQUOTE(JSON_EXTRACT({$field}, '$.end'))", [$end_placeholder => $end]);

      // Value overlaps field on the right.
      $and_cond4 = ($connection->condition('AND'))
        ->where("{$start_placeholder} >= JSON_UNQUOTE(JSON_EXTRACT({$field}, '$.start'))", [$start_placeholder => $start])
        ->where("{$start_placeholder} <= JSON_UNQUOTE(JSON_EXTRACT({$field}, '$.end'))", [$start_placeholder => $start]);

      $or_cond2 = ($connection->condition('OR'))
        ->condition($and_cond3)
        ->condition($and_cond4);

      $or_cond3 = ($connection->condition('OR'))
        ->condition($or_cond1)
        ->condition($or_cond2);

      $query->addWhere($group, $or_cond3);
    }
  }

}
