<?php

namespace Drupal\drpw_field\Plugin\Field\FieldWidget;

use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Field\WidgetBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\daterangepickerwidget\DateRangePickerTrait;
use Drupal\Core\Field\Attribute\FieldWidget;
use Drupal\Core\StringTranslation\TranslatableMarkup;

/**
 * Plugin implementation of the 'daterangepicker_default' widget.
 */
#[FieldWidget(
    id: 'daterangepicker_default',
    label: new TranslatableMarkup('jQuery UI Date Range Picker Default'),
    description: new TranslatableMarkup('Provides a jQuery UI widget similar to the date range picker used in Google Analytics'),
    field_types: ['daterangepicker'],
    multiple_values: FALSE,
)]
class DateRangePickerDefaultWidget extends WidgetBase {

  use DateRangePickerTrait;

  /**
   * {@inheritdoc}
   */
  public function formElement(FieldItemListInterface $items, $delta, array $element, array &$form, FormStateInterface $form_state) {
    $element += [
      '#type' => 'textfield',
      '#default_value' => $items[$delta]->value ?? NULL,
      '#attributes' => [
        'class' => ['daterangepicker'],
        'autocomplete' => 'off',
      ],
      '#attached' => [
        'library' => [
          'daterangepickerwidget/jquery-ui-daterangepicker',
        ],
        'drupalSettings' => [
          'daterangepicker' => [],
        ],
      ],
    ];

    $config = $this->getDateRangePickerConfiguration();
    // Date format of the value used inside
    // the JSON ({ "start": "...", "end": "..." }) submitted by the widget.
    // This needs to be a format recognised by JavaScript's `Date.parse()`.
    // NOTE: DO NOT change this value since we will use this format for the
    // internal storage format.
    $config['alt_format'] = 'yy-mm-dd';

    // Determine the field identifier.
    $form_id = $form_state->getBuildInfo()['form_id'];
    if ($form_id === 'field_config_edit_form' && isset($form['#parents']) && $form['#parents'][0] === 'default_value_input') {
      // We are in the default value field config form.
      $daterangepicker_field_name = $form['#parents'][0] . '[' . $items->getName() . '][' . $delta . '][value]';
    }
    else {
      // We are in a node edit form.
      $daterangepicker_field_name = $items->getName() . '[' . $delta . '][value]';
    }

    $element['#attached']['drupalSettings']['daterangepicker'][$daterangepicker_field_name] = [];
    static::setJavascriptApiOptions($element['#attached']['drupalSettings']['daterangepicker'][$daterangepicker_field_name], $config);

    return ['value' => $element];
  }

  /**
   * {@inheritdoc}
   */
  public static function defaultSettings() {
    return static::getDateRangePickerDefaultOptions() + parent::defaultSettings();
  }

  /**
   * {@inheritdoc}
   */
  public function settingsForm(array $form, FormStateInterface $form_state) {
    $form = parent::settingsForm($form, $form_state);

    $default_values = $this->getDateRangePickerConfiguration();
    static::buildDateRangePickerOptionsForm($form, $default_values);
    $form['#element_validate'] = [
        [$this, 'validateSettingsForm'],
    ];

    return $form;
  }

  /**
   * A custom validator for daterangepicker settings.
   */
  public function validateSettingsForm(array $form, FormStateInterface $form_state) {
    // NOTE: An erroneous field will highlight but for the error message to show
    // we need to enable "inline_form_errors" module.
    $daterangepicker_values = $form_state->getValue($form['#parents']);
    static::validateDateRangePickerOptions($form, $form_state, $daterangepicker_values);
  }

  /**
   * {@inheritdoc}
   */
  public function settingsSummary() {
    $summary = parent::settingsSummary();
    $config = $this->getDateRangePickerConfiguration();
    return array_merge($summary, static::getSummary($config));
  }

  /**
   * Get the Date Range Picker configuration array.
   */
  protected function getDateRangePickerConfiguration() {
    $config = [];
    $options = array_keys(static::getDateRangePickerDefaultOptions());
    foreach ($options as $option) {
      $config[$option] = $this->getSetting($option);
      if (is_numeric($config[$option])) {
        settype($config[$option], 'integer');
      }
    }
    return $config;
  }

}
