<?php

namespace Drupal\drpw_field\Plugin\views\filter;

use Drupal\Core\Form\FormStateInterface;
use Drupal\daterangepickerwidget\DateRangePickerTrait;
use Drupal\views\Plugin\views\filter\FilterPluginBase;
use Drupal\views\Attribute\ViewsFilter;
use Drupal\drpw_field\DateRangePickerViewsOperatorPluginManager;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Filter to handle filtering for the daterangepicker field type.
 *
 * @ingroup views_filter_handlers
 */
#[ViewsFilter("views_daterangepicker_filter")]
class DateRangePickerFilter extends FilterPluginBase {

  use DateRangePickerTrait;

  /**
   * The daterangepicker_views_operator plugin manager.
   *
   * @var \Drupal\drpw_field\DateRangePickerViewsOperatorPluginManager
   */
  protected readonly DateRangePickerViewsOperatorPluginManager $drpOperatorManager;

  /**
   * List of daterangepicker views operators.
   *
   * @var array
   */
  protected $drpOperators;

  /**
   * {@inheritdoc}
   */
  public function __construct(array $configuration, $plugin_id, $plugin_definition, DateRangePickerViewsOperatorPluginManager $drp_operator_manager) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);

    $this->drpOperatorManager = $drp_operator_manager;
    $this->buildOperators();
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
        $configuration,
        $plugin_id,
        $plugin_definition,
        $container->get('plugin.manager.daterangepicker_views_operator'),
      );
  }

  /**
   * Build a list of daterangepicker views operators from available plugins.
   */
  protected function buildOperators() {
    $this->drpOperators = [];
    foreach ($this->drpOperatorManager->getDefinitions() as $plugin_id => $definition) {
      $this->drpOperators[$plugin_id] = [
        'title' => $definition['label'],
        'description' => $definition['description'],
        'short' => $definition['short_name'],
        'values' => $definition['values'],
      ];
    }
  }

  /**
   * {@inheritdoc}
   */
  protected function defineOptions() {
    $options = parent::defineOptions();

    $options['operator'] = ['default' => 'daterangepicker_views_operator_between'];
    $options['daterangepicker'] = ['default' => static::getDateRangePickerDefaultOptions()];

    return $options;
  }

  /**
   * {@inheritdoc}
   */
  public function buildExposeForm(&$form, FormStateInterface $form_state) {
    parent::buildExposeForm($form, $form_state);

    // We do not allow multiple selections with this widget.
    $form['expose']['multiple']['#access'] = FALSE;

    // Build subform to configure daterangepicker config when widget is exposed.
    $form['daterangepicker'] = [
      '#type' => 'details',
      '#title' => $this->t('Date Range Picker options'),
    ];
    $daterangepicker_config = $this->getDateRangePickerConfiguration();
    static::buildDateRangePickerOptionsForm($form['daterangepicker'], $daterangepicker_config);
  }

  /**
   * {@inheritdoc}
   */
  public function validateExposeForm($form, FormStateInterface $form_state) {
    parent::validateExposeForm($form, $form_state);

    $daterangepicker_values = $form_state->getValue(['options', 'daterangepicker']);
    static::validateDateRangePickerOptions($form['daterangepicker'], $form_state, $daterangepicker_values);
  }

  /**
   * Return configurations for the widget.
   */
  protected function getDateRangePickerConfiguration() {
    $config = [];
    $options = array_keys(static::getDateRangePickerDefaultOptions());
    foreach ($options as $option) {
      // Use the same option value irrespective of whether widget is exposed.
      $config[$option] = $this->options['daterangepicker'][$option];
      if (is_numeric($config[$option])) {
        settype($config[$option], 'integer');
      }
    }

    return $config;
  }

  /**
   * Build strings from the operators for 'select' options.
   */
  public function operatorOptions($which = 'title') {
    $options = [];
    foreach ($this->drpOperators as $id => $info) {
      $options[$id] = $info[$which];
    }

    return $options;
  }

  /**
   * {@inheritdoc}
   */
  public function adminSummary() {
    if ($this->isAGroup()) {
      return $this->t('grouped');
    }
    elseif ($this->isExposed()) {
      return $this->t('exposed');
    }

    $options = $this->operatorOptions('short');
    $output = '';
    if (!empty($options[$this->operator])) {
      $output = $options[$this->operator];
    }
    if (in_array($this->operator, $this->operatorValues(1))) {
      $output .= ' ' . $this->value;
    }
    return $output;
  }

  /**
   * {@inheritdoc}
   */
  protected function operatorValues($values = 1) {
    $options = [];
    foreach ($this->drpOperators as $id => $info) {
      if (isset($info['values']) && $info['values'] == $values) {
        $options[] = $id;
      }
    }

    return $options;
  }

  /**
   * {@inheritdoc}
   */
  protected function valueForm(&$form, FormStateInterface $form_state) {
    $form['value'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Value'),
      '#size' => 30,
      '#default_value' => $this->value,
      '#attributes' => [
        'class' => ['daterangepicker'],
        'autocomplete' => 'off',
      ],
      '#attached' => [
        'library' => [
          'daterangepickerwidget/jquery-ui-daterangepicker',
        ],
        'drupalSettings' => [
          'daterangepicker' => [],
        ],
      ],
    ];

    $field_name_map = [
      'views_ui_config_item_form' => 'options[value]',
      'views_exposed_form' => $this->options['expose']['identifier'],
    ];
    $daterangepicker_field_name = $field_name_map[$form_state->getBuildInfo()['form_id']] ?? NULL;
    $daterangepicker_config = $this->getDateRangePickerConfiguration();
    $daterangepicker_config['alt_format'] = 'yy-mm-dd';
    $form['value']['#attached']['drupalSettings']['daterangepicker'][$daterangepicker_field_name] = [];
    static::setJavascriptApiOptions($form['value']['#attached']['drupalSettings']['daterangepicker'][$daterangepicker_field_name], $daterangepicker_config);
  }

  /**
   * {@inheritdoc}
   */
  protected function valueValidate($form, FormStateInterface $form_state) {
    parent::valueValidate($form, $form_state);

    $value = $form_state->getValue(['options', 'value']);
    if (!$this->isExposed() && empty($value)) {
      $form_state->setError($form['value'], $this->t('Value is required.'));
    }
  }

  /**
   * {@inheritdoc}
   */
  protected function buildExposedFiltersGroupForm(&$form, FormStateInterface $form_state) {
    parent::buildExposedFiltersGroupForm($form, $form_state);

    $daterangepicker_config = $this->getDateRangePickerConfiguration();
    $daterangepicker_config['alt_format'] = 'yy-mm-dd';

    // Transform each of the value fields to daterangepicker widgets.
    foreach ($form['group_info']['group_items'] as $item_id => $item) {
      $form['group_info']['group_items'][$item_id]['value']['#attributes'] = [
        'class' => ['daterangepicker'],
        'autocomplete' => 'off',
      ];
      $form['group_info']['group_items'][$item_id]['value']['#attached'] = [
        'library' => [
          'daterangepickerwidget/jquery-ui-daterangepicker',
        ],
        'drupalSettings' => [
          'daterangepicker' => [],
        ],
      ];

      $daterangepicker_field_name = "options[group_info][group_items][{$item_id}][value]";
      $form['group_info']['group_items'][$item_id]['value']['#attached']['drupalSettings']['daterangepicker'][$daterangepicker_field_name] = [];
      static::setJavascriptApiOptions($form['group_info']['group_items'][$item_id]['value']['#attached']['drupalSettings']['daterangepicker'][$daterangepicker_field_name], $daterangepicker_config);
    }
  }

  /**
   * {@inheritdoc}
   */
  public function query() {
    $this->ensureMyTable();
    $field = "$this->tableAlias.$this->realField";

    // Execute the operator logic defined in the plugin.
    if ($this->drpOperatorManager->hasDefinition($this->operator)) {
      $plugin = $this->drpOperatorManager->createInstance($this->operator);
      $plugin->method($this->query, $this->options['group'], $this->position, $this->options, $field, $this->value);
    }
  }

}
