<?php

namespace Drupal\drpw_field\Plugin\views\sort;

use Drupal\Core\Form\FormStateInterface;
use Drupal\views\Attribute\ViewsSort;
use Drupal\views\Plugin\views\sort\SortPluginBase;

/**
 * Basic sort handler for daterangepicker field type.
 */
#[ViewsSort("views_daterangepicker_sort")]
class DateRangePickerSort extends SortPluginBase {

  /**
   * {@inheritdoc}
   */
  protected function defineOptions() {
    $options = parent::defineOptions();

    $options['daterange_part'] = ['default' => 'start_date'];

    return $options;
  }

  /**
   * {@inheritdoc}
   */
  public function buildOptionsForm(&$form, FormStateInterface $form_state) {
    parent::buildOptionsForm($form, $form_state);

    if ($this->options['exposed']) {
      // Hide the order selection component.
      // order will be selected by end user.
      $form['order']['#access'] = FALSE;
    }

    $form['daterange_part'] = [
      '#type' => 'radios',
      '#title' => $this->t('Date range part'),
      '#options' => [
        'start_date' => $this->t('Start date'),
        'end_date' => $this->t('End date'),
        'interval' => $this->t('Range interval'),
      ],
      '#description' => $this->t('Whether to sort by the start date, end date or number of days in a range.'),
      '#default_value' => $this->options['daterange_part'],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function adminSummary() {
    if (!empty($this->options['exposed'])) {
      return $this->t('Exposed');
    }

    $params = [];
    $params[] = $this->options['daterange_part'];
    $params[] = strtolower($this->options['order']);

    return implode(', ', $params);
  }

  /**
   * {@inheritdoc}
   */
  public function query() {
    $this->ensureMyTable();
    $field = "{$this->tableAlias}.{$this->realField}";

    if ($this->options['daterange_part'] !== 'interval') {
      $json_part = $this->options['daterange_part'] === 'start_date' ? '$.start' : '$.end';
      $formula = "JSON_UNQUOTE(JSON_EXTRACT({$field}, '{$json_part}'))";
    }
    else {
      $start_date_formula = "JSON_UNQUOTE(JSON_EXTRACT({$field}, '$.start'))";
      $end_date_formula = "JSON_UNQUOTE(JSON_EXTRACT({$field}, '$.end'))";
      $formula = "DATEDIFF({$end_date_formula}, {$start_date_formula})";
    }

    // Add an expression to the query and orderby its alias.
    $this->query->addOrderBy(NULL, $formula, $this->options['order'], $this->tableAlias . '_' . $this->field . '_' . $this->options['daterange_part']);
  }

}
