<?php

namespace Drupal\daterangepickerwidget;

use Drupal\Component\Serialization\Json;
use Drupal\Core\Form\FormStateInterface;

/**
 * Helper class for JQuery UI Date Range Picker.
 */
trait DateRangePickerTrait {

  /**
   * Return a list of default options used in the Drupal API.
   *
   * @see https://tamble.github.io/jquery-ui-daterangepicker
   * @see https://api.jqueryui.com/datepicker
   *
   * @return array
   *   An array containing the default options.
   */
  public static function getDateRangePickerDefaultOptions(): array {
    return [
      'initial_text' => t('Select date range...'),
      'apply_button_text' => t('Apply'),
      'clear_button_text' => t('Clear'),
      'cancel_button_text' => t('Cancel'),
      'range_splitter' => ' - ',
      'date_format' => 'd M, yy',
      'number_of_months' => 2,
      'show_week' => FALSE,
      'max_date' => '+0d',
      'min_date' => NULL,
      'month_dropdown' => FALSE,
      'year_dropdown' => FALSE,
      'first_day' => 0,
      'step_months' => 1,
      'year_range' => 'c-10:c+10',
      'disable_preset_ranges' => FALSE,
    ];
  }

  /**
   * Map Drupal API options to equivalent Javascript API options.
   *
   * @return array
   *   An array containing the equivalent Javascript API options.
   */
  protected static function mapDrupalToJavascriptOptions(): array {
    return [
      'initial_text' => 'initialText',
      'apply_button_text' => 'applyButtonText',
      'clear_button_text' => 'clearButtonText',
      'cancel_button_text' => 'cancelButtonText',
      'range_splitter' => 'rangeSplitter',
      'date_format' => 'dateFormat',
      // This option is deliberately omitted from
      // ::getDateRangePickerDefaultOptions() because we have a specific
      // storage format.
      'alt_format' => 'altFormat',
      // This option is deliberately omitted from
      // ::getDateRangePickerDefaultOptions(). However, preset ranges can be
      // set in the form element or with hook_form_alter.
      'preset_ranges' => 'presetRanges',
      'number_of_months' => ['datepickerOptions' => 'numberOfMonths'],
      'show_week' => ['datepickerOptions' => 'showWeek'],
      'max_date' => ['datepickerOptions' => 'maxDate'],
      'min_date' => ['datepickerOptions' => 'minDate'],
      'month_dropdown' => ['datepickerOptions' => 'changeMonth'],
      'year_dropdown' => ['datepickerOptions' => 'changeYear'],
      'first_day' => ['datepickerOptions' => 'firstDay'],
      'step_months' => ['datepickerOptions' => 'stepMonths'],
      'year_range' => ['datepickerOptions' => 'yearRange'],
      'disable_preset_ranges' => 'presetRanges',
      // This is a pseudo option which is used to set the default range in JS.
      'default_value' => 'defaultValue',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public static function buildDateRangePickerOptionsForm(array &$form, array $default_values): void {
    $form['initial_text'] = [
      '#type' => 'textfield',
      '#title' => t('Initial text'),
      '#default_value' => $default_values['initial_text'],
    ];
    $form['apply_button_text'] = [
      '#type' => 'textfield',
      '#title' => t('Apply button text'),
      '#default_value' => $default_values['apply_button_text'],
    ];
    $form['clear_button_text'] = [
      '#type' => 'textfield',
      '#title' => t('Clear button text'),
      '#default_value' => $default_values['clear_button_text'],
    ];
    $form['cancel_button_text'] = [
      '#type' => 'textfield',
      '#title' => t('Cancel button text'),
      '#default_value' => $default_values['cancel_button_text'],
    ];
    $form['range_splitter'] = [
      '#type' => 'textfield',
      '#title' => t('Range splitter'),
      '#default_value' => $default_values['range_splitter'],
    ];
    $form['date_format'] = [
      '#type' => 'textfield',
      '#title' => t('Date format'),
      '#description' => t('See <a href="https://api.jqueryui.com/datepicker/#utility-formatDate" target="_blank">the documentation for JQuery UI datepicker formats</a>.'),
      '#default_value' => $default_values['date_format'],
    ];
    $form['number_of_months'] = [
      '#type' => 'number',
      '#title' => t('Number of months'),
      '#default_value' => $default_values['number_of_months'],
    ];
    $form['show_week'] = [
      '#type' => 'checkbox',
      '#title' => t('Show week'),
      '#default_value' => $default_values['show_week'],
    ];
    $form['max_date'] = [
      '#type' => 'textfield',
      '#title' => t('Max date'),
      '#description' => t('See <a href="https://api.jqueryui.com/datepicker/#option-maxDate" target="_blank">the documentation</a>.'),
      '#default_value' => $default_values['max_date'],
    ];
    $form['min_date'] = [
      '#type' => 'textfield',
      '#title' => t('Min date'),
      '#description' => t('See <a href="https://api.jqueryui.com/datepicker/#option-minDate" target="_blank">the documentation</a>.'),
      '#default_value' => $default_values['min_date'],
    ];
    $form['month_dropdown'] = [
      '#type' => 'checkbox',
      '#title' => t('Enable month selector'),
      '#description' => t('Whether the month selector must be rendered as a dropdown instead of plain text.'),
      '#default_value' => $default_values['month_dropdown'],
    ];
    $form['year_dropdown'] = [
      '#type' => 'checkbox',
      '#title' => t('Enable year selector'),
      '#description' => t('Whether the year selector must be rendered as a dropdown instead of plain text.'),
      '#default_value' => $default_values['year_dropdown'],
    ];
    $form['first_day'] = [
      '#type' => 'number',
      '#title' => t('First day of week'),
      '#description' => t('Set the first day of the week: Sunday is 0, Monday is 1, etc.'),
      '#min' => 0,
      '#max' => 6,
      '#step' => 1,
      '#default_value' => $default_values['first_day'],
    ];
    $form['step_months'] = [
      '#type' => 'number',
      '#title' => t('Step months'),
      '#description' => t('Set how many months to move when clicking the previous/next links.'),
      '#min' => 1,
      '#max' => 12,
      '#step' => 1,
      '#default_value' => $default_values['step_months'],
    ];
    $form['year_range'] = [
      '#type' => 'textfield',
      '#title' => t('Year range'),
      '#description' => t('See <a href="https://api.jqueryui.com/datepicker/#option-yearRange" target="_blank">the documentation</a>.'),
      '#default_value' => $default_values['year_range'],
    ];
    $form['disable_preset_ranges'] = [
      '#type' => 'checkbox',
      '#title' => t('Disable preset ranges'),
      '#description' => t('Whether to hide the preset ranges column in the date picker.'),
      '#default_value' => $default_values['disable_preset_ranges'],
    ];
  }

  /**
   * Get summary of the data array.
   *
   * @param array $data
   *   The data array.
   *
   * @return array
   *   The summary of the data.
   */
  public static function getSummary(array $data): array {
    $summary = [];
    foreach ($data as $option => $value) {
      $value_display = $value;

      if (is_null($value) || $value === '') {
        $value_display = 'NULL';
      }
      elseif (is_bool($value)) {
        $value_display = $value ? 'TRUE' : 'FALSE';
      }

      $summary[] = t('@option: @value', [
        '@option' => ucwords(str_replace('_', ' ', $option)),
        '@value' => $value_display,
      ]);
    }

    return $summary;
  }

  /**
   * Map Drupal API options to Javascript API options and update drupalSettings.
   *
   * @param array $drupalSettings
   *   The Drupal settings array.
   * @param array $data
   *   The data array.
   */
  public static function setJavascriptApiOptions(array &$drupalSettings, array $data): void {
    $optionsMap = static::mapDrupalToJavascriptOptions();
    foreach ($data as $option => $value) {
      if ($option === 'disable_preset_ranges' && $value) {
        $value = Json::encode([]);
      }
      elseif ($option === 'disable_preset_ranges' && !$value) {
        continue;
      }
      static::setJavascriptApiOption($drupalSettings, $optionsMap[$option], $value);
    }
  }

  /**
   * Set the JavaScript API option in the drupalSettings array.
   *
   * @param array $drupalSettings
   *   The Drupal settings array.
   * @param string|array $key
   *   The key for the settings.
   * @param mixed $value
   *   The value to be set.
   */
  protected static function setJavascriptApiOption(array &$drupalSettings, $key, $value): void {
    if (is_array($key)) {
      if (!isset($drupalSettings[key($key)])) {
        $drupalSettings[key($key)] = [];
      }
      static::setJavascriptApiOption($drupalSettings[key($key)], current($key), $value);
    }
    else {
      $drupalSettings[$key] = $value;
    }
  }

  /**
   * Validation helper for daterangepicker options.
   *
   * @param array $form
   *   The fragment of the original form (i.e. subform) that contains
   *   the daterangepicker fields.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current state of the form.
   * @param array $values
   *   Daterangepicker values extracted from $form_state.
   */
  public function validateDateRangePickerOptions($form, FormStateInterface $form_state, array $values) {
    // Validate month step counter.
    if ($values['step_months'] > $values['number_of_months']) {
      $form_state->setError($form['step_months'], $this->t('step_months cannot be greater than number_of_months.'));
    }
  }

}
