<?php

namespace Drupal\daterangepickerwidget\Element;

use Drupal\Component\Serialization\Json;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Render\Element\FormElementBase;
use Drupal\Core\Render\Attribute\FormElement;
use Drupal\Core\Render\Element;
use Drupal\daterangepickerwidget\DateRangePickerTrait;

/**
 * Provides a jQuery UI Date Range Picker widget form element.
 */
#[FormElement('daterangepicker')]
class DateRangePickerElement extends FormElementBase {

  use DateRangePickerTrait;

  /**
   * {@inheritdoc}
   */
  public function getInfo() {
    $options = [];
    foreach (static::getDateRangePickerDefaultOptions() as $option => $value) {
      $options['#' . $option] = $value;
    }

    $class = static::class;
    return [
      '#input' => TRUE,
      '#empty_value' => '',
      '#default_value' => '',
      '#pre_render' => [
        [$class, 'preRenderDateRangePicker'],
      ],
      '#process' => [
        [$class, 'processDateRangePicker'],
      ],
      '#element_validate' => [
        [$class, 'validateDateRangePicker'],
      ],
    ] + $options;
  }

  /**
   * Prepares a #type 'daterangepicker' render element for input.html.twig.
   *
   * @param array $element
   *   An associative array containing the properties of the element.
   *
   * @return array
   *   The $element with prepared variables ready for input.html.twig.
   */
  public static function preRenderDateRangePicker($element) {
    $element['#attributes']['type'] = 'text';
    Element::setAttributes($element, ['id', 'name', 'value']);
    static::setAttributes($element, ['form-text', 'form-daterangepicker']);

    return $element;
  }

  /**
   * Process the Date Range Picker element.
   */
  public static function processDateRangePicker(&$element, FormStateInterface $form_state, &$complete_form) {
    static::validateElementSettings($element);

    $element_name = $element['#name'];
    $element[$element_name] = [
      '#type' => 'textfield',
      '#title' => $element['#title'] ?? '',
      '#title_display' => $element['#title_display'] ?? '',
      '#description' => $element['#description'] ?? '',
      '#default_value' => $element['#default_value'],
      '#value' => $element['#value'],
      // Set required to FALSE so that the widget validator
      // (::validateDateRangePicker) can take effect and we can show the error
      // message.
      '#required' => FALSE,
      '#attributes' => $element['#attributes'],
      '#attached' => [
        'library' => [
          'daterangepickerwidget/jquery-ui-daterangepicker',
        ],
        'drupalSettings' => [
          'daterangepicker' => [],
        ],
      ],
    ];

    // Manipulate classes on the text field (input element)
    $classes = $element[$element_name]['#attributes']['class'] ?? [];
    $element[$element_name]['#attributes']['class'] = array_merge(
        $classes,
        [
          'daterangepicker',
          'form-daterangepicker',
        ]
    );

    // Manipulate classes for the text field label.
    if (isset($element['#required']) && $element['#required']) {
      $classes = $element[$element_name]['#label_attributes']['class'] ?? [];
      $element[$element_name]['#label_attributes']['class'] = array_merge(
          $classes,
          [
            'js-form-required',
            'form-required',
          ]
      );
    }

    $element[$element_name]['#id'] = $element['#id'];
    $element[$element_name]['#name'] = $element['#name'];

    $config = static::getDateRangePickerConfiguration($element);
    // "alt_format" should not be altered by the user since "yy-mm-dd"
    // is the internal storage format.
    $config['alt_format'] = 'yy-mm-dd';

    // Check whether the form element defined custom preset ranges.
    if (isset($element['#preset_ranges'])) {
      $config['preset_ranges'] = Json::encode($element['#preset_ranges']);
    }

    if (isset($element['#default_value']) && is_array($element['#default_value'])) {
      $config['default_value'] = $element['#default_value'];
    }

    // Store widget configuration corresponding to this element.
    $element[$element_name]['#attached']['drupalSettings']['daterangepicker'][$element_name] = [];
    static::setJavascriptApiOptions($element[$element_name]['#attached']['drupalSettings']['daterangepicker'][$element_name], $config);

    return $element;
  }

  /**
   * Custom validator to validate settings of a daterangepicker form element.
   */
  public static function validateElementSettings(array $element) {
    if ($element['#step_months'] > $element['#number_of_months']) {
      throw new \Exception('step_months cannot be greater than number_of_months.');
    }
  }

  /**
   * Validate the Date Range Picker element.
   */
  public static function validateDateRangePicker(&$element, FormStateInterface $form_state, &$complete_form) {
    $value = $form_state->getValue($element['#name']);
    if ($element['#required'] && empty($value)) {
      $form_state->setError($element, t('@field_title is required', ['@field_title' => $element['#title'] ?? $element['#name']]));
      $form_state->setValueForElement($element, '');
    }
  }

  /**
   * {@inheritdoc}
   */
  public static function valueCallback(&$element, $input, FormStateInterface $form_state) {
    $ret = $element['#default_value'] ?? '';
    if ($input !== FALSE) {
      $ret = Json::decode($input);
    }

    return $ret;
  }

  /**
   * Get the Date Range Picker configuration array.
   */
  protected static function getDateRangePickerConfiguration(array $element) {
    $config = [];
    $options = array_keys(static::getDateRangePickerDefaultOptions());
    foreach ($options as $option) {
      $config[$option] = $element['#' . $option];
      if (is_numeric($config[$option])) {
        settype($config[$option], 'integer');
      }
    }

    return $config;
  }

}
