<?php

namespace Drupal\db_cleanup\Drush\Commands;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drush\Attributes as CLI;
use Drush\Commands\DrushCommands;
use Drupal\Core\Database\Connection;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Symfony\Component\Console\Helper\ProgressBar;
use Symfony\Component\Console\Helper\Table;

class DbCleanupCommands extends DrushCommands {

  use StringTranslationTrait;

  /**
   * @var \Drupal\Core\Database\Connection
   */
  protected $database;

  /**
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * DbCleanupCommands constructor.
   *
   * @param \Drupal\Core\Database\Connection $database
   */
  public function __construct(Connection $database, EntityTypeManagerInterface $entity_type_manager) {
    parent::__construct();
    $this->database = $database;
    $this->entityTypeManager = $entity_type_manager;
  }

  /**
   * Check database tables fragmentation.
   */
  #[CLI\Command(name: 'db:check-fragmentation', aliases: ['db:cf'])]
  #[CLI\Usage(name: 'drush db:check-fragmentation', description: 'Check database tables fragmentation.')]
  public function checkFragmentation() {
    $innodbFilePerTableEnabled = $this->checkInnodbFilePerTable();

    if (!$innodbFilePerTableEnabled) {
      $this->logger()
        ->error($this->t("innodb_file_per_table parameter is OFF or can\'t be checked. Can't check fragmentation."));
      return;
    }

    $results = $this->getFragmentedTables();
    if (empty($results)) {
      $this->logger()->notice($this->t('No fragmented table has been found.'));
      return;
    }

    $this->output()->writeln('');
    $this->output()->writeln('Fragmented tables:');
    $this->output()->writeln('');

    $header = [
      'Table',
      'Data size (MB)',
      'Index size (MB)',
      'Free space (MB)',
      'Fragmentation ratio (%)',
    ];

    $rows = [];
    $total_free_space = 0;
    foreach ($results as $result) {
      $total_free_space = $total_free_space + $result->data_free;
      // Sets the color according to the fragmentation ratio
      $fragRatio = round($result->frag_ratio, 2);
      if ($fragRatio < 10) {
        $coloredRatio = "<fg=green>$fragRatio</fg=green>";
      }
      elseif ($fragRatio < 25) {
        $coloredRatio = "<fg=yellow>$fragRatio</fg=yellow>";
      }
      else {
        $coloredRatio = "<fg=red>$fragRatio</fg=red>";
      }

      $rows[] = [
        $result->table_name,
        $result->data_length,
        $result->index_length,
        $result->data_free,
        $coloredRatio,
      ];
    }

    // Display results
    $this->io()->table($header, $rows);
    $this->logger()
      ->notice($this->t('You can reclaim a maximum of @total_space MB.', ['@total_space' => $total_free_space]));
    $this->logger()
      ->notice($this->t('To reclaim space, you can use the db:cleanup-tables drush command.'));
  }

  /**
   * Cleanup fragmented database tables.
   */
  #[CLI\Command(name: 'db:cleanup-tables', aliases: ['db:opt'])]
  #[CLI\Usage(name: 'drush db:cleanup-tables', description: 'Cleanup fragmented database tables.')]
  #[CLI\Option(name: 'threshold', description: 'Fragmentation ratio threshold to cleanup tables (0-100).')]
  #[CLI\Option(name: 'dry-run', description: 'Displays tables that would be cleaned up without running the cleanup.')]
  public function cleanup_tables($options = [
    'threshold' => 10,
    'dry-run' => FALSE,
  ]
  ) {
    $innodb_file_per_table_enabled = $this->checkInnodbFilePerTable();
    if (!$innodb_file_per_table_enabled) {
      $this->logger()
        ->error($this->t("innodb_file_per_table parameter is OFF or can\'t be checked. Can't reclaim database space"));
      return;
    }

    $threshold = $options['threshold'];
    if (!is_numeric($threshold) || $threshold < 0 || $threshold > 100) {
      $this->logger()
        ->error($this->t('Threshold must be a number between 0 and 100.'));
      return;
    }

    $dry_run = $options['dry-run'];

    $results = $this->getFragmentedTables($threshold);

    if (empty($results)) {
      $this->logger()
        ->notice('No table above the fragmentation threshold has been found.');
      return;
    }

    if ($dry_run) {
      $this->output()
        ->writeln('Dry run - following tables would be cleaned up:');
      foreach ($results as $result) {
        $frag_ratio = round($result->frag_ratio, 2);
        if ($frag_ratio < 10) {
          $color = 'green';
        }
        elseif ($frag_ratio < 25) {
          $color = 'yellow';
        }
        else {
          $color = 'red';
        }
        $this->output()
          ->writeln(sprintf('- <fg=cyan>%s</fg=cyan> (fragmentation ratio: <fg=%s>%s%%</fg=%s>)', $result->table_name, $color, $frag_ratio, $color));
      }
      return;
    }

    $total_database_size = $this->getTotalDatabaseSize();
    $this->output()->writeln("Current database size: $total_database_size MB");
    $this->output()->writeln('Starting tables cleanup...');
    $cleaned_tables = 0;
    $error_tables_count = 0;
    foreach ($results as $result) {
      $table_name = $result->table_name;
      $this->output()
        ->write($this->t('Optimizing <fg=cyan>@table_name</fg=cyan>... ', ['@table_name' => $table_name]));
      try {
        $this->database->query("ALTER TABLE {$table_name} ENGINE=InnoDB");
        $this->output()->writeln('<fg=green>OK</fg=green>');
        $cleaned_tables++;
      }
      catch (\Exception $e) {
        $this->output()->writeln('<fg=red>ERROR</fg=red>');
        $this->logger()
          ->error($this->t('An error occurred while optimizing @table_name table : @message',
            ['@table_name' => $table_name, '@message' => $e->getMessage()]));
        $error_tables_count++;
      }
    }

    $this->output()->writeln('');
    $this->logger()
      ->success($this->t('@cleaned_tables tables cleaned up successfully.',
        ['@cleaned_tables' => $cleaned_tables]));
    if ($error_tables_count > 0) {
      $this->logger()
        ->error($this->t('@error_count tables couldn\'t be cleaned up.',
          ['@error_count' => $error_tables_count]));
    }
    $new_database_size = $this->getTotalDatabaseSize();
    $freed_space = $total_database_size - $new_database_size;
    $this->output()->writeln("Total space freed : $freed_space MB");
    $this->output()->writeln("New database size: $new_database_size");
  }

  protected function checkInnodbFilePerTable() {
    $result = $this->database->query("SHOW VARIABLES LIKE 'innodb_file_per_table'")
      ->fetchAssoc();
    if (!$result) {
      return FALSE;
    }
    return strtolower($result['Value']) === 'on';
  }

  protected function getFragmentedTables($threshold = 0) {
    $query = "SELECT TABLE_NAME as table_name,
              ROUND(DATA_LENGTH/1024/1024, 2) as data_length,
              ROUND(INDEX_LENGTH/1024/1024, 2) as index_length,
              ROUND(DATA_FREE/1024/1024, 2) as data_free,
              (DATA_FREE/(INDEX_LENGTH+DATA_LENGTH)*100) as frag_ratio
              FROM information_schema.tables
              WHERE DATA_FREE > 0
              AND ENGINE = 'InnoDB'
              HAVING frag_ratio >= :threshold
              ORDER BY frag_ratio";

    return $this->database->query($query, [':threshold' => $threshold])
      ->fetchAll();
  }

  protected function getTotalDatabaseSize() {
    $total_db_size_query = "SELECT ROUND(SUM(data_length + index_length) / 1024 / 1024, 2) AS total_size
            FROM
              information_schema.TABLES
            GROUP BY
              table_schema
            HAVING
              table_schema = DATABASE()";
    $total_database_size = $this->database->query($total_db_size_query)->fetchField();
    return $total_database_size;
  }

  #[CLI\Command(name: 'db:check-db-size', aliases: ['db:cs'])]
  #[CLI\Usage(name: 'db:check-db-size', description: 'Check database size and biggest tables.')]
  #[CLI\Help(description: 'Check total database size and biggest tables.')]
  #[CLI\Option(name: 'number', description: 'Number of tables to check.')]
  public function checkTablesSize($options = [
    'number' => 30,
  ]
  ) {
    $total_database_size = $this->getTotalDatabaseSize();
    $tables_to_check = $options['number'];
    // Query to get the size of the biggest tables in the database
    $query = "SELECT
              table_name,
              ROUND(((data_length + index_length) / 1024 / 1024), 2) AS size_mb,
              ROUND(DATA_FREE/1024/1024, 2) as data_free
            FROM
              information_schema.TABLES
            WHERE
              table_schema = DATABASE()
            ORDER BY
              size_mb DESC
            LIMIT $tables_to_check";

    $results = $this->database->query($query)->fetchAll();

    if (empty($results)) {
      $this->logger()->notice($this->t('No tables found in the database.'));
      return;
    }

    $this->output()
      ->writeln($this->t('@tables_number biggest tables in the database:', ['@tables_number' => $tables_to_check]));
    $this->output()
      ->writeln('-----------------------------------------------------------');

    $results = array_reverse($results);
    $has_large_revision_table = FALSE;
    foreach ($results as $result) {
      $table_name = $result->table_name;
      $size_mb = $result->size_mb;
      $data_free = $result->data_free;

      // Determine color based on size
      if ($size_mb >= 50) {
        // Red for tables ≥ 50 MB
        $color = 'red';
        // Check if the table is a revision table
        if (!$has_large_revision_table && str_contains($table_name, 'revision')) {
          $has_large_revision_table = TRUE;
        }
      }
      elseif ($size_mb >= 10) {
        // Yellow for tables > 10 MB
        $color = 'yellow';
      }
      else {
        // Green for tables < 10 MB
        $color = 'green';
      }
      $output = "$table_name <fg=$color>$size_mb MB</fg=$color>";
      if ($data_free > 0) {
        $output .= " ($data_free MB empty space)";
      }
      $this->output()
        ->writeln($output);
    }
    $this->output()->writeln('');
    if ($has_large_revision_table) {
      $this->output()
        ->writeln($this->t('It seems the revision tables are taking a lot of space. You can get more info on revisions with the command <fg=cyan>drush db:check-node-revisions</>'));
    }

    $this->output()->writeln('');
    $this->output()->writeln($this->t("Total size of the database: <fg=cyan>@total_size MB</fg=cyan>", ['@total_size' => $total_database_size]));
  }

  #[CLI\Command(name: 'db:check-node-revisions', aliases: ['db:cnr'])]
  #[CLI\Option(name: 'limit', description: 'Number of top nodes with most revisions to display')]
  #[CLI\Usage(name: 'db:check-node-revisions', description: 'Show revision statistics and the 10 nodes with most revisions.')]
  #[CLI\Usage(name: 'db:check-node-revisions --limit=5', description: 'Show revision statistics and only the top 5 nodes with most revisions.')]
  #[CLI\Help(description: 'Displays statistics about node revisions, including average, median, and nodes with the most revisions.')]
  public function check_revisions($options = [
    'limit' => 10, // Number of top nodes to display
  ]
  ) {
    $limit = $options['limit'];
    if (!is_numeric($limit) || $limit < 1) {
      $this->logger()
        ->error($this->t('Limit must be a positive number.'));
      return;
    }

    // Get total node count
    $total_nodes_query = "SELECT COUNT(DISTINCT nid) as count FROM {node}";
    $total_nodes = $this->database->query($total_nodes_query)->fetchField();

    if ($total_nodes == 0) {
      $this->logger()->notice($this->t('No nodes found in the database.'));
      return;
    }

    $total_revision_query = "SELECT SUM(revision_count) FROM (SELECT COUNT(r.vid) as revision_count
                           FROM {node_field_data} n
                           LEFT JOIN {node_revision} r ON n.nid = r.nid
                           GROUP BY n.nid) total_revisions";

    // Calculate average revisions per node
    $total_revisions = $this->database->query($total_revision_query)->fetchField();

    $average_revisions_query = "SELECT AVG(revision_count) FROM (SELECT COUNT(r.vid) as revision_count
                           FROM {node_field_data} n
                           LEFT JOIN {node_revision} r ON n.nid = r.nid
                           GROUP BY n.nid) average_revisions";

    $average_revisions = $this->database->query($average_revisions_query)->fetchField();
    $average_revisions = round($average_revisions, 2);

    // Get nodes with most revisions
    $top_nodes_query = "SELECT n.nid, n.title, n.type as content_type, COUNT(r.vid) as revision_count,
                        DATE_FORMAT(FROM_UNIXTIME(created), '%d/%m/%Y') as created_date, n.langcode
                     FROM {node_field_data} n
                     LEFT JOIN {node_revision} r ON n.nid = r.nid
                     GROUP BY n.nid
                     ORDER BY revision_count DESC
                     LIMIT $limit";

    $top_nodes = $this->database->query($top_nodes_query)->fetchAll();

    // Output the statistics
    $this->output()->writeln('');
    $this->output()->writeln($this->t('Node Revision Statistics:'));
    $this->output()->writeln('--------------------------');
    $this->output()
      ->writeln($this->t('Total nodes: @count', ['@count' => $total_nodes]));
    $this->output()
      ->writeln($this->t('Total revisions: @count', ['@count' => $total_revisions]));
    $this->output()
      ->writeln($this->t('Average revisions per node: @avg', ['@avg' => $average_revisions]));
    $this->output()->writeln('');

    // Output top nodes with most revisions
    $this->output()
      ->writeln($this->t('Top @count Nodes with most revisions:', ['@count' => $limit]));

    if (empty($top_nodes)) {
      $this->output()->writeln($this->t('No nodes with revisions found.'));
    }
    else {
      $table_header = [
        $this->t('Node ID'),
        $this->t('Content Type'),
        $this->t('Title'),
        $this->t('Revisions'),
        $this->t('Created Date'),
        $this->t('Language'),
      ];

      $table_rows = [];
      foreach ($top_nodes as $node) {
        // Truncate long titles
        $title = strlen($node->title) > 80 ? substr($node->title, 0, 77) . '...' : $node->title;

        $table_rows[] = [
          $node->nid,
          $node->content_type,
          $title,
          $node->revision_count,
          $node->created_date,
          $node->langcode,
        ];
      }

      $table = new Table($this->output());
      $table->setHeaders($table_header)
        ->setRows($table_rows);
      $table->render();
    }
  }

}
