<?php

namespace Drupal\db_cleanups\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Database\Database;

/**
 * Configuration form for DB Cleanup module.
 */
class DbCleanupSettingsForm extends ConfigFormBase {

  protected function getEditableConfigNames() {
    return ['db_cleanups.settings'];
  }

  public function getFormId() {
    return 'db_cleanups_settings_form';
  }

  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config('db_cleanups.settings');

    $form['watchdog_interval'] = [
      '#type' => 'number',
      '#title' => $this->t('Watchdog Cleanup Interval (hours)'),
      '#default_value' => $config->get('watchdog_interval') ?? 1,
      '#min' => 1,
      '#description' => $this->t('How often to clear the watchdog table.'),
    ];

    $form['cache_interval'] = [
      '#type' => 'number',
      '#title' => $this->t('Cache Tables Cleanup Interval (hours)'),
      '#default_value' => $config->get('cache_interval') ?? 8,
      '#min' => 1,
      '#description' => $this->t('How often to clear cache_* tables.'),
    ];

    $form['optimize_tables'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Optimize tables after cleanup'),
      '#default_value' => $config->get('optimize_tables') ?? TRUE,
      '#description' => $this->t('Run OPTIMIZE TABLE after cleanup to reclaim space and improve performance.'),
    ];

    $form['manual_cleanup'] = [
      '#type' => 'submit',
      '#value' => $this->t('Run Cleanup Now'),
      '#submit' => ['::runCleanupNow'],
      '#button_type' => 'primary',
    ];

    return parent::buildForm($form, $form_state);
  }

  public function submitForm(array &$form, FormStateInterface $form_state) {
    $this->config('db_cleanups.settings')
      ->set('watchdog_interval', $form_state->getValue('watchdog_interval'))
      ->set('cache_interval', $form_state->getValue('cache_interval'))
      ->set('optimize_tables', $form_state->getValue('optimize_tables'))
      ->save();

    parent::submitForm($form, $form_state);
    $this->messenger()->addStatus($this->t('Settings saved successfully.'));
  }

  public function runCleanupNow(array &$form, FormStateInterface $form_state) {
    $config = $this->config('db_cleanups.settings');
    $optimize = (bool) $config->get('optimize_tables', TRUE);

    $before_size = $this->getDatabaseSizeMB();

    db_cleanups_clear_watchdog($optimize);
    db_cleanups_clear_cache_tables($optimize);

    $after_size = $this->getDatabaseSizeMB();
    $saved = max(0, $before_size - $after_size);

    $this->messenger()->addStatus($this->t('Cleanup completed successfully.'));
    if ($optimize) {
      $this->messenger()->addMessage($this->t('Database tables optimized after cleanup.'));
    }

    $this->messenger()->addMessage($this->t(
      '📊 Database size before cleanup: @before MB<br>📉 After cleanup: @after MB<br>💾 Space saved: @saved MB',
      [
        '@before' => number_format($before_size, 2),
        '@after' => number_format($after_size, 2),
        '@saved' => number_format($saved, 2),
      ]
    ), 'status');
  }

  private function getDatabaseSizeMB() {
    $connection = Database::getConnection();
    $db_name = $connection->getConnectionOptions()['database'];

    $query = $connection->query(
      "SELECT SUM(data_length + index_length) / 1024 / 1024 AS size
       FROM information_schema.TABLES
       WHERE table_schema = :schema",
      [':schema' => $db_name]
    );

    $result = $query->fetchObject();
    return $result && $result->size ? (float) $result->size : 0.0;
  }

}
