<?php

namespace Drupal\db_health\Form;

use Drupal\Core\Database\Connection;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\TempStore\PrivateTempStoreFactory;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 *
 */
class DbHealthLogsForm extends FormBase {

  protected $database;
  protected $tempStore;
  protected $confirming = FALSE;

  public function __construct(Connection $database, PrivateTempStoreFactory $tempstore_factory) {
    $this->database = $database;
    $this->tempStore = $tempstore_factory->get('db_health');
  }

  /**
   *
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('database'),
      $container->get('tempstore.private')
    );
  }

  /**
   *
   */
  public function getFormId() {
    return 'db_health_manage_logs_form';
  }

  /**
   *
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $selected = $this->tempStore->get('selected_timestamps') ?? [];

    foreach ($selected as $key => $timestamp) {
      $formatted = \Drupal::service('date.formatter')->format($timestamp, 'short');
      $selected[$key] = $this->t('@ts', ['@ts' => $formatted, '@raw' => $timestamp]);
    }

    // If confirmation step.
    if ($form_state->get('confirm')) {
      $this->confirming = TRUE;

      $form['message'] = [
        '#markup' => $this->t('Are you sure you want to delete all entries with the following timestamps?'),
      ];

      $form['selected_timestamps'] = [
        '#markup' => '<ul><li>' . implode('</li><li>', $selected) . '</li></ul>',
      ];

      $form['confirm'] = [
        '#type' => 'submit',
        '#value' => $this->t('Confirm Delete'),
      ];

      $form['cancel'] = [
        '#type' => 'submit',
        '#value' => $this->t('Cancel'),
        '#submit' => ['::cancel'],
      ];

      return $form;
    }

    // Load distinct timestamps.
    $query = $this->database->select('db_health_sizes', 'p');
    $query->distinct();
    $query->fields('p', ['timestamp']);
    $result = $query->execute()->fetchCol();

    if (empty($result)) {
      $form['empty'] = ['#markup' => $this->t('No logs found.')];
      return $form;
    }

    $options = array_combine($result, $result);

    foreach ($options as $key => $timestamp) {
      $formatted = \Drupal::service('date.formatter')->format($timestamp, 'short');
      $options[$key] = $this->t('@ts', ['@ts' => $formatted, '@raw' => $timestamp]);
    }

    $form['timestamps'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Select timestamps to delete'),
      '#options' => $options,
    ];

    $form['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Delete'),
    ];

    return $form;
  }

  /**
   *
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    if ($this->confirming) {
      $selected = $this->tempStore->get('selected_timestamps') ?? [];

      if (!empty($selected)) {
        $this->database->delete('db_health_sizes')
          ->condition('timestamp', $selected, 'IN')
          ->execute();

        $this->messenger()->addStatus($this->t('Deleted logs for selected timestamps.'));
        $this->tempStore->delete('selected_timestamps');
      }
    }
    else {
      $values = array_filter($form_state->getValue('timestamps'));
      if (empty($values)) {
        $this->messenger()->addWarning($this->t('No timestamps selected.'));
        return;
      }

      $this->tempStore->set('selected_timestamps', $values);
      $form_state->setRebuild(TRUE);
      $form_state->set('confirm', TRUE);
    }
  }

  /**
   *
   */
  public function cancel(array &$form, FormStateInterface $form_state) {
    $this->tempStore->delete('selected_timestamps');
    $form_state->setRebuild(TRUE);
    $form_state->set('confirm', FALSE);
  }

}
