<?php

namespace Drupal\db_health\Form;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\Url;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides the settings form for the DB Health module.
 */
class DbHealthSettingsForm extends FormBase {
  use StringTranslationTrait;

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The messenger service.
   *
   * @var \Drupal\Core\Messenger\MessengerInterface
   */
  protected $messenger;

  /**
   * Constructs a new \Drupal\db_health\Form\DbHealthSettingsForm object.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   * @param \Drupal\Core\Messenger\MessengerInterface $messenger
   *   The messenger service.
   */
  public function __construct(ConfigFactoryInterface $config_factory, MessengerInterface $messenger) {
    $this->configFactory = $config_factory;
    $this->messenger = $messenger;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('config.factory'),
      $container->get('messenger')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'db_health_settings_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $form['cron_notice'] = [
      '#type' => 'markup',
      '#markup' => '<div class="messages messages--warning"><strong>' . $this->t('Important:') . '</strong> ' . $this->t("The execution of this job depends on Drupal's cron system. Make sure that <strong>Drupal cron is running frequently enough</strong> (e.g., via Drush or external cron) to ensure this task runs as scheduled. You can configure it at <a href=\":url\">Cron settings</a>.", [
        ':url' => Url::fromRoute('system.cron_settings')->toString(),
      ]) . '</div>',
    ];

    $intervals = [
      300 => $this->t('Every 5 minutes'),
      600 => $this->t('Every 10 minutes'),
      900 => $this->t('Every 15 minutes'),
      1800 => $this->t('Every 30 minutes'),
      3600 => $this->t('1 hour'),
      10800 => $this->t('3 hours'),
      21600 => $this->t('6 hours'),
      43200 => $this->t('12 hours'),
      86400 => $this->t('1 day'),
      604800 => $this->t('1 week'),
      1209600 => $this->t('2 weeks'),
      2592000 => $this->t('1 month'),
    ];

    $config = $this->configFactory->getEditable('db_health.settings');

    $form['interval'] = [
      '#type' => 'select',
      '#title' => $this->t('Cron Interval'),
      '#description' => $this->t('This module collects information from the database at the interval you select above.'),
      '#options' => $intervals,
      '#default_value' => $config->get('interval') ?: 43200,
    ];

    $form['submit'] = [
      '#type' => 'submit',
      '#attributes' => [
        'style' => 'margin:0',
      ],
      '#value' => $this->t('Save'),
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $config = $this->configFactory->getEditable('db_health.settings');
    $config->set('interval', $form_state->getValue('interval'))
      ->save();
    $this->messenger->addMessage($this->t('Settings saved.'));
  }

}
