<?php

namespace Drupal\db_health;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Database\Connection;

/**
 * Checks and stores paragraph table sizes.
 */
class DbHealthSizeChecker {

  /**
   * The database connection.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected $database;

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * Constructs a new \Drupal\db_health\DbHealthSizeChecker object.
   *
   * @param \Drupal\Core\Database\Connection $database
   *   The database connection.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   */
  public function __construct(Connection $database, ConfigFactoryInterface $config_factory) {
    $this->database = $database;
    $this->configFactory = $config_factory;
  }

  /**
   * Runs the checker and stores sizes.
   */
  public function run(): void {
    $tables = \Drupal::database()->query("SHOW TABLE STATUS")->fetchAllAssoc('Name');
    // Now returns actual table names.
    $tables_used = $this->getTables();
    $timestamp = time();
    $tables_arr = [];

    foreach ($tables as $name => $info) {
      $size = (int) $info->Data_length + (int) $info->Index_length;
      $count = (int) \Drupal::database()->query("SELECT COUNT(*) FROM {{$name}}")->fetchField();

      if ($name !== 'db_health_sizes' && $name !== 'db_health_table_names' && $name != 'db') {
        $table_id = $this->getOrCreateTableId($name);
        \Drupal::database()->insert('db_health_sizes')
          ->fields([
            'table_id' => $table_id,
            'size' => $size,
            'count' => $count,
            'timestamp' => $timestamp,
          ])->execute();

        $tables_arr[] = $name;
      }
    }

    // Handle tables that existed before but are now missing.
    $diff = array_diff($tables_used, $tables_arr);
    if ($diff) {
      foreach ($diff as $table_name) {
        $table_id = $this->getOrCreateTableId($table_name);
        if ($table_name !== 'db_health_sizes' && $table_name !== 'db_health_table_names' && $table_name != 'db') {
          \Drupal::database()->insert('db_health_sizes')
            ->fields([
              'table_id' => $table_id,
              'size' => 0,
              'count' => 0,
              'timestamp' => $timestamp,
            ])->execute();
        }
      }
    }

    // Log full DB size as a pseudo-table "db".
    $query = \Drupal::database()->query("
      SELECT ROUND(SUM(data_length + index_length), 2) AS size_mb 
      FROM information_schema.tables 
      WHERE table_schema = :schema",
      [':schema' => \Drupal::database()->getConnectionOptions()['database']]
    );
    $size = $query->fetchField();
    $table_id = $this->getOrCreateTableId('db');

    $total_rows = \Drupal::database()->query("
      SELECT SUM(TABLE_ROWS)
      FROM information_schema.tables
      WHERE table_schema = :schema",
      [':schema' => \Drupal::database()->getConnectionOptions()['database']]
    )->fetchField();

    \Drupal::database()->insert('db_health_sizes')
      ->fields([
        'table_id' => $table_id,
        'size' => $size,
        'count' => $total_rows,
        'timestamp' => $timestamp,
      ])->execute();
  }

  /**
   * Lists all size records with table names.
   *
   * @return array
   *   An array of size records with table names.
   */
  public function list() {
    $query = \Drupal::database()->select('db_health_sizes', 'phs')
      ->fields('phs', ['size', 'timestamp'])
      ->fields('t', ['table_name'])
      ->orderBy('timestamp', 'ASC');
    $query->innerJoin('db_health_table_names', 't', 'phs.table_id = t.id');

    return $query->execute()->fetchAll();
  }

  /**
   * Get distinct paragraph table names used.
   *
   * @return array
   *   An array of table names.
   */
  private function getTables() {
    $query = \Drupal::database()->select('db_health_sizes', 'phs');
    $query->innerJoin('db_health_table_names', 't', 'phs.table_id = t.id');
    $query->addField('t', 'table_name');
    $result = $query->execute()->fetchCol();

    $result = array_unique($result);

    return $result;
  }

  /**
   * Get or create table_id by table name.
   *
   * @param string $table_name
   *   The table name.
   *
   * @return int
   *   The table ID.
   */
  private function getOrCreateTableId(string $table_name): int {
    $id = \Drupal::database()->query("SELECT id FROM {db_health_table_names} WHERE table_name = :name", [
      ':name' => $table_name,
    ])->fetchField();

    if (!$id) {
      $id = \Drupal::database()->insert('db_health_table_names')
        ->fields(['table_name' => $table_name])
        ->execute();
    }

    return (int) $id;
  }

  /**
   * Gets the total size of all module tables.
   *
   * @return float|int
   *   The total size in MB.
   */
  public function getModuleTablesSize() {
    $query = \Drupal::database()->query("
      SELECT ROUND(SUM(data_length + index_length), 2) AS size_mb 
      FROM information_schema.tables 
      WHERE table_schema = :schema ",
      [':schema' => \Drupal::database()->getConnectionOptions()['database']]
    );
    $size = $query->fetchField();
    return $size;
  }

}
